/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.gjc.spi;

import javax.resource.ResourceException;
import javax.resource.spi.ConnectionRequestInfo;
import com.sun.gjc.spi.ConnectionManager;
import com.sun.gjc.util.SecurityUtils;
import javax.resource.spi.security.PasswordCredential;
import java.sql.DriverManager;
import com.sun.gjc.common.DataSourceSpec;
import com.sun.gjc.common.DataSourceObjectBuilder;
import java.sql.SQLException;
import java.sql.Connection;
import javax.sql.PooledConnection;

import com.sun.logging.*;
import java.util.logging.Logger;
import java.util.logging.Level;

import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;

/**
 * <code>ManagedConnectionFactory</code> implementation for Generic JDBC Connector.
 * This class is extended by the DataSource specific <code>ManagedConnection</code> factories
 * and the <code>ManagedConnectionFactory</code> for the <code>DriverManager</code>.
 *
 * @version	1.0, 02/08/03
 * @author	Evani Sai Surya Kiran, Aditya Gore
 */

public abstract class ManagedConnectionFactory implements javax.resource.spi.ManagedConnectionFactory, 
    javax.resource.spi.ValidatingManagedConnectionFactory, java.io.Serializable {
    
    protected DataSourceSpec spec = new DataSourceSpec();
    protected transient DataSourceObjectBuilder dsObjBuilder;
    
    protected java.io.PrintWriter logWriter = null;
    protected javax.resource.spi.ResourceAdapter ra = null;
    
    private static Logger _logger;
    static {
        _logger = LogDomains.getLogger( LogDomains.RSR_LOGGER );
    }
    protected javax.resource.spi.LazyEnlistableConnectionManager cm_;
    protected boolean isLazyCm_;
    
    /**
     * Creates a Connection Factory instance. The <code>ConnectionManager</code> implementation
     * of the resource adapter is used here.
     * 
     * @return	Generic JDBC Connector implementation of <code>javax.sql.DataSource</code>
     */
    public Object createConnectionFactory() {
        if(logWriter != null) {
            logWriter.println("In createConnectionFactory()");
        }
        com.sun.gjc.spi.DataSource cf = new com.sun.gjc.spi.DataSource(
            (javax.resource.spi.ManagedConnectionFactory)this, null);
        
        return cf;
    }
    
    /**
     * Creates a Connection Factory instance. The <code>ConnectionManager</code> implementation
     * of the application server is used here.
     * 
     * @param	cxManager	<code>ConnectionManager</code> passed by the application server
     * @return	Generic JDBC Connector implementation of <code>javax.sql.DataSource</code>
     */
    public Object createConnectionFactory(javax.resource.spi.ConnectionManager cxManager) {
        if(logWriter != null) {
            logWriter.println("In createConnectionFactory(javax.resource.spi.ConnectionManager cxManager)");
        }
        com.sun.gjc.spi.DataSource cf = new com.sun.gjc.spi.DataSource(
            (javax.resource.spi.ManagedConnectionFactory)this, cxManager);
        
        if ( cxManager instanceof javax.resource.spi.LazyEnlistableConnectionManager ) {
            cm_ = (javax.resource.spi.LazyEnlistableConnectionManager)cxManager;
            isLazyCm_ = true;
        }
        return cf; 
    }
    
    /**
     * Creates a new physical connection to the underlying EIS resource
     * manager.
     * 
     * @param	subject	<code>Subject</code> instance passed by the application server
     * @param	cxRequestInfo	<code>ConnectionRequestInfo</code> which may be created
     *       	             	as a result of the invocation <code>getConnection(user, password)</code>
     *       	             	on the <code>DataSource</code> object
     * @return	<code>ManagedConnection</code> object created
     * @throws	ResourceException	if there is an error in instantiating the
     *        	                 	<code>DataSource</code> object used for the
     *       				creation of the <code>ManagedConnection</code> object
     * @throws	SecurityException	if there ino <code>PasswordCredential</code> object
     *        	                 	satisfying this request 
     * @throws	ResourceAllocationException	if there is an error in allocating the 
     *						physical connection
     */
    public abstract javax.resource.spi.ManagedConnection createManagedConnection(javax.security.auth.Subject subject,
        ConnectionRequestInfo cxRequestInfo) throws ResourceException;
    
    /**
     * Check if this <code>ManagedConnectionFactory</code> is equal to 
     * another <code>ManagedConnectionFactory</code>.
     * 
     * @param	other	<code>ManagedConnectionFactory</code> object for checking equality with
     * @return	true	if the property sets of both the 
     *			<code>ManagedConnectionFactory</code> objects are the same
     *		false	otherwise
     */
    public abstract boolean equals(Object other);
    
    /**
     * Get the log writer for this <code>ManagedConnectionFactory</code> instance.
     *
     * @return	<code>PrintWriter</code> associated with this <code>ManagedConnectionFactory</code> instance
     * @see	<code>setLogWriter</code>
     */
    public java.io.PrintWriter getLogWriter() {
        return logWriter;
    }
    
    /**
     * Get the <code>ResourceAdapter</code> for this <code>ManagedConnectionFactory</code> instance.
     *
     * @return	<code>ResourceAdapter</code> associated with this <code>ManagedConnectionFactory</code> instance
     * @see	<code>setResourceAdapter</code>
     */
    public javax.resource.spi.ResourceAdapter getResourceAdapter() {
        if(logWriter != null) {
            logWriter.println("In getResourceAdapter");
        }
        return ra;
    }
    
    /**
     * Returns the hash code for this <code>ManagedConnectionFactory</code>.
     *
     * @return	hash code for this <code>ManagedConnectionFactory</code>
     */
    public int hashCode(){
        if(logWriter != null) {
                logWriter.println("In hashCode");
        }
        return spec.hashCode();
    }
    
    /**
     * Returns a matched <code>ManagedConnection</code> from the candidate 
     * set of <code>ManagedConnection</code> objects.
     * 
     * @param	connectionSet	<code>Set</code> of  <code>ManagedConnection</code>
     *				objects passed by the application server
     * @param	subject	 passed by the application server
     *			for retrieving information required for matching
     * @param	cxRequestInfo	<code>ConnectionRequestInfo</code> passed by the application server
     *				for retrieving information required for matching
     * @return	<code>ManagedConnection</code> that is the best match satisfying this request
     * @throws	ResourceException	if there is an error accessing the <code>Subject</code>
     *					parameter or the <code>Set</code> of <code>ManagedConnection</code>
     *					objects passed by the application server
     */
    public javax.resource.spi.ManagedConnection matchManagedConnections(java.util.Set connectionSet, 
        javax.security.auth.Subject subject, ConnectionRequestInfo cxRequestInfo) throws ResourceException {
        if(logWriter != null) {
            logWriter.println("In matchManagedConnections");
        }
        
        if(connectionSet == null) {
            return null;
        }
        
        PasswordCredential pc = SecurityUtils.getPasswordCredential(this, subject, cxRequestInfo);
        
        java.util.Iterator iter = connectionSet.iterator();
        com.sun.gjc.spi.ManagedConnection mc = null;
        while(iter.hasNext()) {
            try {
                mc = (com.sun.gjc.spi.ManagedConnection) iter.next();
            } catch(java.util.NoSuchElementException nsee) {
	        _logger.log(Level.SEVERE, "jdbc.exc_iter");
                throw new ResourceException(nsee.getMessage());
            }
            if(pc == null && this.equals(mc.getManagedConnectionFactory())) {
		return mc;
            } else if(SecurityUtils.isPasswordCredentialEqual(pc, mc.getPasswordCredential()) == true) {
		return mc;
            }
        }
        return null;
    }
    
   /**
     * This method returns a set of invalid <code>ManagedConnection</code> 
     * objects chosen from a specified set of <code>ManagedConnection</code>
     * objects.
     *
     * @param connectionSet a set of <code>ManagedConnection</code> objects
     * that need to be validated.
     *
     * @return a set of invalid <code>ManagedConnection</code> objects.
     *
     * @throws ResourceException generic exception.
     */
    public Set getInvalidConnections(Set connectionSet) throws ResourceException {
        Iterator iter = connectionSet.iterator();
        Set invalid = new HashSet();
        while( iter.hasNext() ) {
            ManagedConnection mc = (ManagedConnection) iter.next();
            try {
                isValid( mc );
            } catch( ResourceException re ) {
                invalid.add( mc );
                mc.connectionErrorOccurred( re, null );
        
            }
        }
	return invalid;
    }
								  
    //GJCINT
    /**
     * Checks if a <code>ManagedConnection</code> is to be validated or not
     * and validates it or returns.
     * 
     * @param	mc	<code>ManagedConnection</code> to be validated
     * @throws	ResourceException	if the connection is not valid or
     *        	          		if validation method is not proper
     */
    void isValid(com.sun.gjc.spi.ManagedConnection mc) throws ResourceException {

        if(mc.isTransactionInProgress()) {
	    return;
        }
    
	String conVal = spec.getDetail( DataSourceSpec.CONNECTIONVALIDATIONREQUIRED );
	
        boolean connectionValidationRequired = 
            ( conVal == null ) ? false : (new Boolean(conVal.toLowerCase())).booleanValue();
        if( connectionValidationRequired == false || mc == null) {
            return;
        }
        
        
        String validationMethod = spec.getDetail(DataSourceSpec.VALIDATIONMETHOD).toLowerCase();
        
        mc.checkIfValid();
        /**
         * The above call checks if the actual physical connection
         * is usable or not.
         */
        java.sql.Connection con = mc.getActualConnection();
        
        if(validationMethod.equals("auto-commit") == true) {
            isValidByAutoCommit(con);
        } else if(validationMethod.equalsIgnoreCase("meta-data") == true) {
            isValidByMetaData(con);
        } else if(validationMethod.equalsIgnoreCase("table") == true) {
            isValidByTableQuery(con, spec.getDetail(DataSourceSpec.VALIDATIONTABLENAME));
        } else {
            throw new ResourceException("The validation method is not proper");
        }
    }
    
    /**
     * Checks if a <code>java.sql.Connection</code> is valid or not
     * by checking its auto commit property.
     * 
     * @param	con	<code>java.sql.Connection</code> to be validated
     * @throws	ResourceException	if the connection is not valid
     */
    protected void isValidByAutoCommit(java.sql.Connection con) throws ResourceException {
        if(con == null) {
            throw new ResourceException("The connection is not valid as "
                + "the connection is null");
        }
        
        try {
           // Notice that using something like 
           // dbCon.setAutoCommit(dbCon.getAutoCommit()) will cause problems with
           // some drivers like sybase
           // We do not validate connections that are already enlisted 
	   //in a transaction 
	   // We cycle autocommit to true and false to by-pass drivers that 
	   // might cache the call to set autocomitt
           // Also notice that some XA data sources will throw and exception if
           // you try to call setAutoCommit, for them this method is not recommended

           boolean ac = con.getAutoCommit();
           if (ac) {
                con.setAutoCommit(false);
           } else {
                con.rollback(); // prevents uncompleted transaction exceptions
                con.setAutoCommit(true);
           }
        
	   con.setAutoCommit(ac);

        } catch(Exception sqle) {
	    _logger.log(Level.SEVERE, "jdbc.exc_autocommit");
            throw new ResourceException(sqle.getMessage());
        }
    }
    
    /**
     * Checks if a <code>java.sql.Connection</code> is valid or not
     * by checking its meta data.
     * 
     * @param	con	<code>java.sql.Connection</code> to be validated
     * @throws	ResourceException	if the connection is not valid
     */
    protected void isValidByMetaData(java.sql.Connection con) throws ResourceException {
        if(con == null) {
            throw new ResourceException("The connection is not valid as "
                + "the connection is null");
        }
        
        try {
            java.sql.DatabaseMetaData dmd = con.getMetaData();
        } catch(Exception sqle) {
	    _logger.log(Level.SEVERE, "jdbc.exc_md");
            throw new ResourceException("The connection is not valid as "
                + "getting the meta data failed: " + sqle.getMessage());
        }
    }
    
    /**
     * Checks if a <code>java.sql.Connection</code> is valid or not
     * by querying a table.
     * 
     * @param	con	<code>java.sql.Connection</code> to be validated
     * @param	tableName	table which should be queried
     * @throws	ResourceException	if the connection is not valid
     */
    protected void isValidByTableQuery(java.sql.Connection con, 
        String tableName) throws ResourceException {
        if(con == null) {
            throw new ResourceException("The connection is not valid as "
                + "the connection is null");
        }
	
        java.sql.Statement stmt = null;
	java.sql.ResultSet rs = null;
        try {
            stmt = con.createStatement();
            rs = stmt.executeQuery("SELECT COUNT(*) FROM " + tableName);
        } catch(Exception sqle) {
	    _logger.log(Level.SEVERE, "jdbc.exc_execute");
            throw new ResourceException("The connection is not valid as "
                + "querying the table " + tableName + " failed: " + sqle.getMessage());
        } finally {
	    try {
	        if (rs != null ) {
	            rs.close();
	        }
	    } catch (Exception e1) {}
	
    	    try {
	        if (stmt != null ) {
	            stmt.close();
	        }
	    } catch (Exception e2) {}
	}
    }
    
    /**
     * Sets the isolation level specified in the <code>ConnectionRequestInfo</code>
     * for the <code>ManagedConnection</code> passed.
     *
     * @param	mc	<code>ManagedConnection</code>
     * @throws	ResourceException	if the isolation property is invalid
     *					or if the isolation cannot be set over the connection
     */
    protected void setIsolation(com.sun.gjc.spi.ManagedConnection mc) throws ResourceException {
    	
    	java.sql.Connection con = mc.getActualConnection();
    	if(con == null) {
    	    return;
    	}
    	
    	String tranIsolation = spec.getDetail(DataSourceSpec.TRANSACTIONISOLATION);
    	if(tranIsolation != null && tranIsolation.equals("") == false) {
    	    int tranIsolationInt = getTransactionIsolationInt(tranIsolation);
    	    try {
    	        con.setTransactionIsolation(tranIsolationInt);
    	    } catch(java.sql.SQLException sqle) {
	        _logger.log(Level.SEVERE, "jdbc.exc_tx_level");
    	        throw new ResourceException("The transaction isolation could "
    	            + "not be set: " + sqle.getMessage());
    	    }
    	}
    }
    
    /**
     * Resets the isolation level for the <code>ManagedConnection</code> passed.
     * If the transaction level is to be guaranteed to be the same as the one
     * present when this <code>ManagedConnection</code> was created, as specified
     * by the <code>ConnectionRequestInfo</code> passed, it sets the transaction
     * isolation level from the <code>ConnectionRequestInfo</code> passed. Else,
     * it sets it to the transaction isolation passed.
     *
     * @param	mc	<code>ManagedConnection</code>
     * @param	tranIsol	int
     * @throws	ResourceException	if the isolation property is invalid
     *					or if the isolation cannot be set over the connection
     */
    void resetIsolation(com.sun.gjc.spi.ManagedConnection mc, int tranIsol) throws ResourceException {
    	
    	java.sql.Connection con = mc.getActualConnection();
    	if(con == null) {
    	    return;
    	}
    	
    	String tranIsolation = spec.getDetail(DataSourceSpec.TRANSACTIONISOLATION);
    	if(tranIsolation != null && tranIsolation.equals("") == false) {
    	    String guaranteeIsolationLevel = spec.getDetail(DataSourceSpec.GUARANTEEISOLATIONLEVEL);
    	    
    	    if(guaranteeIsolationLevel != null && guaranteeIsolationLevel.equals("") == false) {
    	        boolean guarantee = (new Boolean(guaranteeIsolationLevel.toLowerCase())).booleanValue();
    	        
    	        if(guarantee) {
    	            int tranIsolationInt = getTransactionIsolationInt(tranIsolation);
    	            try {
    	                if(tranIsolationInt != con.getTransactionIsolation()) {
    	                    con.setTransactionIsolation(tranIsolationInt);
    	                }
    	            } catch(java.sql.SQLException sqle) {
	                _logger.log(Level.SEVERE, "jdbc.exc_tx_level");
    	                throw new ResourceException("The isolation level could not be set: "
    	                    + sqle.getMessage());
    	            }
    	        } else {
    	            try {
    	                if(tranIsol != con.getTransactionIsolation()) {
    	                    con.setTransactionIsolation(tranIsol);
    	                }
    	            } catch(java.sql.SQLException sqle) {
	                _logger.log(Level.SEVERE, "jdbc.exc_tx_level");
    	                throw new ResourceException("The isolation level could not be set: "
    	                    + sqle.getMessage());
    	            }
    	        }
    	    }
    	}
    }
    
    /**
     * Gets the integer equivalent of the string specifying
     * the transaction isolation.
     *
     * @param	tranIsolation	string specifying the isolation level
     * @return	tranIsolationInt	the <code>java.sql.Connection</code> constant
     *					for the string specifying the isolation.
     */
    private int getTransactionIsolationInt(String tranIsolation) throws ResourceException {
    	if(tranIsolation.equalsIgnoreCase("read-uncommitted")) {
    	    return java.sql.Connection.TRANSACTION_READ_UNCOMMITTED;
    	} else if(tranIsolation.equalsIgnoreCase("read-committed")) {
    	    return java.sql.Connection.TRANSACTION_READ_COMMITTED;
    	} else if(tranIsolation.equalsIgnoreCase("repeatable-read")) {
    	    return java.sql.Connection.TRANSACTION_REPEATABLE_READ;
    	} else if(tranIsolation.equalsIgnoreCase("serializable")) {
    	    return java.sql.Connection.TRANSACTION_SERIALIZABLE;
    	} else {
    	    throw new ResourceException("Invalid transaction isolation; the transaction "
    	        + "isolation level can be empty or any of the following: "
    	            + "read-uncommitted, read-committed, repeatable-read, serializable");
    	}
    }
    
    /** 
     * Common operation performed by all the child MCFs before returning a created mc
     */ 
    protected void validateAndSetIsolation( ManagedConnection mc ) 
            throws ResourceException 
    {
       isValid( mc );
       setIsolation( mc );
    }
    
    /**
     * Set the log writer for this <code>ManagedConnectionFactory</code> instance.
     *
     * @param	out	<code>PrintWriter</code> passed by the application server
     * @see	<code>getLogWriter</code>
     */
    public void setLogWriter(java.io.PrintWriter out) {
        logWriter = out;
    }
    
    /**
     * Set the associated <code>ResourceAdapter</code> JavaBean.
     *
     * @param	ra	<code>ResourceAdapter</code> associated with this 
     *			<code>ManagedConnectionFactory</code> instance
     * @see	<code>getResourceAdapter</code>
     */
    public void setResourceAdapter(javax.resource.spi.ResourceAdapter ra) {
        this.ra = ra;   
    }
    
    /**
     * Sets the user name
     *
     * @param	user	<code>String</code>
     */
    public void setUser(String user) {
        spec.setDetail(DataSourceSpec.USERNAME, user);
    }
    
    /**
     * Gets the user name
     *
     * @return	user
     */
    public String getUser() {
        return spec.getDetail(DataSourceSpec.USERNAME);
    }
    
    /**
     * Sets the user name
     *
     * @param	user	<code>String</code>
     */
    public void setuser(String user) {
        spec.setDetail(DataSourceSpec.USERNAME, user);
    }
    
    /**
     * Gets the user name
     *
     * @return	user
     */
    public String getuser() {
        return spec.getDetail(DataSourceSpec.USERNAME);
    }
    
    /**
     * Sets the password
     *
     * @param	passwd	<code>String</code>
     */
    public void setPassword(String passwd) {
        spec.setDetail(DataSourceSpec.PASSWORD, passwd);
    }
    
    /**
     * Gets the password
     *
     * @return	passwd
     */
    public String getPassword() {
        return spec.getDetail(DataSourceSpec.PASSWORD);
    }
    
    /**
     * Sets the password
     *
     * @param	passwd	<code>String</code>
     */
    public void setpassword(String passwd) {
        spec.setDetail(DataSourceSpec.PASSWORD, passwd);
    }
    
    /**
     * Gets the password
     *
     * @return	passwd
     */
    public String getpassword() {
        return spec.getDetail(DataSourceSpec.PASSWORD);
    }
    
    /**
     * Sets the class name of the data source
     *
     * @param	className	<code>String</code>
     */
    public void setClassName(String className) {
        spec.setDetail(DataSourceSpec.CLASSNAME, className);
    }
    
    /**
     * Gets the class name of the data source
     *
     * @return	className
     */
    public String getClassName() {
        return spec.getDetail(DataSourceSpec.CLASSNAME);
    }
    
    /**
     * Sets the class name of the data source
     *
     * @param	className	<code>String</code>
     */
    public void setclassName(String className) {
        spec.setDetail(DataSourceSpec.CLASSNAME, className);
    }
    
    /**
     * Gets the class name of the data source
     *
     * @return	className
     */
    public String getclassName() {
        return spec.getDetail(DataSourceSpec.CLASSNAME);
    }
    
    /**
     * Sets if connection validation is required or not
     *
     * @param	conVldReq	<code>String</code>
     */
    public void setConnectionValidationRequired(String conVldReq) {
        spec.setDetail(DataSourceSpec.CONNECTIONVALIDATIONREQUIRED, conVldReq);
    }
    
    /**
     * Returns if connection validation is required or not
     *
     * @return	connection validation requirement
     */
    public String getConnectionValidationRequired() {
        return spec.getDetail(DataSourceSpec.CONNECTIONVALIDATIONREQUIRED);
    }
    
    /**
     * Sets if connection validation is required or not
     *
     * @param	conVldReq	<code>String</code>
     */
    public void setconnectionValidationRequired(String conVldReq) {
        spec.setDetail(DataSourceSpec.CONNECTIONVALIDATIONREQUIRED, conVldReq);
    }
    
    /**
     * Returns if connection validation is required or not
     *
     * @return	connection validation requirement
     */
    public String getconnectionValidationRequired() {
        return spec.getDetail(DataSourceSpec.CONNECTIONVALIDATIONREQUIRED);
    }
    
    /**
     * Sets the validation method required
     *
     * @param	validationMethod	<code>String</code>
     */
    public void setValidationMethod(String validationMethod) {
            spec.setDetail(DataSourceSpec.VALIDATIONMETHOD, validationMethod);
    }
    
    /**
     * Returns the connection validation method type
     *
     * @return	validation method
     */
    public String getValidationMethod() {
        return spec.getDetail(DataSourceSpec.VALIDATIONMETHOD);
    }
    
    /**
     * Sets the validation method required
     *
     * @param	validationMethod	<code>String</code>
     */
    public void setvalidationMethod(String validationMethod) {
            spec.setDetail(DataSourceSpec.VALIDATIONMETHOD, validationMethod);
    }
    
    /**
     * Returns the connection validation method type
     *
     * @return	validation method
     */
    public String getvalidationMethod() {
        return spec.getDetail(DataSourceSpec.VALIDATIONMETHOD);
    }
    
    /**
     * Sets the table checked for during validation
     *
     * @param	table	<code>String</code>
     */
    public void setValidationTableName(String table) {
        spec.setDetail(DataSourceSpec.VALIDATIONTABLENAME, table);
    }
    
    /**
     * Returns the table checked for during validation
     *
     * @return	table
     */
    public String getValidationTableName() {
        return spec.getDetail(DataSourceSpec.VALIDATIONTABLENAME);
    }
    
    /**
     * Sets the table checked for during validation
     *
     * @param	table	<code>String</code>
     */
    public void setvalidationTableName(String table) {
        spec.setDetail(DataSourceSpec.VALIDATIONTABLENAME, table);
    }
    
    /**
     * Returns the table checked for during validation
     *
     * @return	table
     */
    public String getvalidationTableName() {
        return spec.getDetail(DataSourceSpec.VALIDATIONTABLENAME);
    }
    
    /**
     * Sets the transaction isolation level
     *
     * @param	trnIsolation	<code>String</code>
     */
    public void setTransactionIsolation(String trnIsolation) {
        spec.setDetail(DataSourceSpec.TRANSACTIONISOLATION, trnIsolation);
    }
    
    /**
     * Returns the transaction isolation level
     *
     * @return	transaction isolation level
     */
    public String getTransactionIsolation() {
        return spec.getDetail(DataSourceSpec.TRANSACTIONISOLATION);
    }
    
    /**
     * Sets the transaction isolation level
     *
     * @param	trnIsolation	<code>String</code>
     */
    public void settransactionIsolation(String trnIsolation) {
        spec.setDetail(DataSourceSpec.TRANSACTIONISOLATION, trnIsolation);
    }
    
    /**
     * Returns the transaction isolation level
     *
     * @return	transaction isolation level
     */
    public String gettransactionIsolation() {
        return spec.getDetail(DataSourceSpec.TRANSACTIONISOLATION);
    }
    
    /**
     * Sets if the transaction isolation level is to be guaranteed
     *
     * @param	guaranteeIsolation	<code>String</code>
     */
    public void setGuaranteeIsolationLevel(String guaranteeIsolation) {
        spec.setDetail(DataSourceSpec.GUARANTEEISOLATIONLEVEL, guaranteeIsolation);
    }
    
    /**
     * Returns the transaction isolation level
     *
     * @return	isolation level guarantee
     */
    public String getGuaranteeIsolationLevel() {
        return spec.getDetail(DataSourceSpec.GUARANTEEISOLATIONLEVEL);
    }
    
    /**
     * Sets if the transaction isolation level is to be guaranteed
     *
     * @param	guaranteeIsolation	<code>String</code>
     */
    public void setguaranteeIsolationLevel(String guaranteeIsolation) {
        spec.setDetail(DataSourceSpec.GUARANTEEISOLATIONLEVEL, guaranteeIsolation);
    }
    
    /**
     * Returns the transaction isolation level
     *
     * @return	isolation level guarantee
     */
    public String getguaranteeIsolationLevel() {
        return spec.getDetail(DataSourceSpec.GUARANTEEISOLATIONLEVEL);
    }
    
    protected boolean isEqual( PasswordCredential pc, String user, 
        String password) {
        
	//if equal get direct connection else 
	//get connection with user and password.
	
	String thisUser = (pc == null) ? null : pc.getUserName();
	char[] passwordArray = (pc == null) ? null : pc.getPassword();
	String thisPassword = (passwordArray == null) ? null : new String(passwordArray);

	return (isStringEqual(thisUser, user) && isStringEqual(thisPassword, password)); 

    }

    private boolean isStringEqual(String str1, String str2) {
	return str1 == null ? str2 == null : str1.equals( str2 );
    }

    
    /**
     * Sets the server name.
     *
     * @param	serverName	<code>String</code>
     * @see	<code>getServerName</code>
     */
    public void setserverName(String serverName) {
        spec.setDetail(DataSourceSpec.SERVERNAME, serverName);
    }
    
    /**
     * Gets the server name.
     *
     * @return	serverName
     * @see	<code>setServerName</code>
     */
    public String getserverName() {
        return spec.getDetail(DataSourceSpec.SERVERNAME);
    }
    
    /**
     * Sets the server name.
     *
     * @param	serverName	<code>String</code>
     * @see	<code>getServerName</code>
     */
    public void setServerName(String serverName) {
        spec.setDetail(DataSourceSpec.SERVERNAME, serverName);
    }
    
    /**
     * Gets the server name.
     *
     * @return	serverName
     * @see	<code>setServerName</code>
     */
    public String getServerName() {
        return spec.getDetail(DataSourceSpec.SERVERNAME);
    }
    
    /**
     * Sets the port number.
     *
     * @param	portNumber	<code>String</code>
     * @see	<code>getPortNumber</code>
     */
    public void setportNumber(String portNumber) {
        spec.setDetail(DataSourceSpec.PORTNUMBER, portNumber);
    }
    
    /**
     * Gets the port number.
     *
     * @return	portNumber
     * @see	<code>setPortNumber</code>
     */
    public String getportNumber() {
        return spec.getDetail(DataSourceSpec.PORTNUMBER);
    }
    
    /**
     * Sets the port number.
     *
     * @param	portNumber	<code>String</code>
     * @see	<code>getPortNumber</code>
     */
    public void setPortNumber(String portNumber) {
        spec.setDetail(DataSourceSpec.PORTNUMBER, portNumber);
    }
    
    /**
     * Gets the port number.
     *
     * @return	portNumber
     * @see	<code>setPortNumber</code>
     */
    public String getPortNumber() {
        return spec.getDetail(DataSourceSpec.PORTNUMBER);
    }
    
    /**
     * Sets the database name.
     *
     * @param	databaseName	<code>String</code>
     * @see	<code>getDatabaseName</code>
     */
    public void setdatabaseName(String databaseName) {
        spec.setDetail(DataSourceSpec.DATABASENAME, databaseName);
    }
    
    /**
     * Gets the database name.
     *
     * @return	databaseName
     * @see	<code>setDatabaseName</code>
     */
    public String getdatabaseName() {
        return spec.getDetail(DataSourceSpec.DATABASENAME);
    }
    
    /**
     * Sets the database name.
     *
     * @param	databaseName	<code>String</code>
     * @see	<code>getDatabaseName</code>
     */
    public void setDatabaseName(String databaseName) {
        spec.setDetail(DataSourceSpec.DATABASENAME, databaseName);
    }
    
    /**
     * Gets the database name.
     *
     * @return	databaseName
     * @see	<code>setDatabaseName</code>
     */
    public String getDatabaseName() {
        return spec.getDetail(DataSourceSpec.DATABASENAME);
    }
    
    /**
     * Sets the data source name.
     *
     * @param	dsn <code>String</code>
     * @see	<code>getDataSourceName</code>
     */
    public void setdataSourceName(String dsn) {
        spec.setDetail(DataSourceSpec.DATASOURCENAME, dsn);
    }
    
    /**
     * Gets the data source name.
     *
     * @return	dsn
     * @see	<code>setDataSourceName</code>
     */
    public String getdataSourceName() {
        return spec.getDetail(DataSourceSpec.DATASOURCENAME);
    }
    
    /**
     * Sets the data source name.
     *
     * @param	dsn <code>String</code>
     * @see	<code>getDataSourceName</code>
     */
    public void setDataSourceName(String dsn) {
        spec.setDetail(DataSourceSpec.DATASOURCENAME, dsn);
    }
    
    /**
     * Gets the data source name.
     *
     * @return	dsn
     * @see	<code>setDataSourceName</code>
     */
    public String getDataSourceName() {
        return spec.getDetail(DataSourceSpec.DATASOURCENAME);
    }
    
    /**
     * Sets the description.
     *
     * @param	desc	<code>String</code>
     * @see	<code>getDescription</code>
     */
    public void setdescription(String desc) {
        spec.setDetail(DataSourceSpec.DESCRIPTION, desc);
    }
    
    /**
     * Gets the description.
     *
     * @return	desc
     * @see	<code>setDescription</code>
     */
    public String getdescription() {
        return spec.getDetail(DataSourceSpec.DESCRIPTION);
    }
    
    /**
     * Sets the description.
     *
     * @param	desc	<code>String</code>
     * @see	<code>getDescription</code>
     */
    public void setDescription(String desc) {
        spec.setDetail(DataSourceSpec.DESCRIPTION, desc);
    }
    
    /**
     * Gets the description.
     *
     * @return	desc
     * @see	<code>setDescription</code>
     */
    public String getDescription() {
        return spec.getDetail(DataSourceSpec.DESCRIPTION);
    }
    
    /**
     * Sets the network protocol.
     *
     * @param	nwProtocol	<code>String</code>
     * @see	<code>getNetworkProtocol</code>
     */
    public void setnetworkProtocol(String nwProtocol) {
        spec.setDetail(DataSourceSpec.NETWORKPROTOCOL, nwProtocol);
    }
    
    /**
     * Gets the network protocol.
     *
     * @return	nwProtocol
     * @see	<code>setNetworkProtocol</code>
     */
    public String getnetworkProtocol() {
        return spec.getDetail(DataSourceSpec.NETWORKPROTOCOL);
    }
    
    /**
     * Sets the network protocol.
     *
     * @param	nwProtocol	<code>String</code>
     * @see	<code>getNetworkProtocol</code>
     */
    public void setNetworkProtocol(String nwProtocol) {
        spec.setDetail(DataSourceSpec.NETWORKPROTOCOL, nwProtocol);
    }
    
    /**
     * Gets the network protocol.
     *
     * @return	nwProtocol
     * @see	<code>setNetworkProtocol</code>
     */
    public String getNetworkProtocol() {
        return spec.getDetail(DataSourceSpec.NETWORKPROTOCOL);
    }
    
    /**
     * Sets the role name.
     *
     * @param	roleName	<code>String</code>
     * @see	<code>getRoleName</code>
     */
    public void setroleName(String roleName) {
        spec.setDetail(DataSourceSpec.ROLENAME, roleName);
    }
    
    /**
     * Gets the role name.
     *
     * @return	roleName
     * @see	<code>setRoleName</code>
     */
    public String getroleName() {
        return spec.getDetail(DataSourceSpec.ROLENAME);
    }
    
    /**
     * Sets the role name.
     *
     * @param	roleName	<code>String</code>
     * @see	<code>getRoleName</code>
     */
    public void setRoleName(String roleName) {
        spec.setDetail(DataSourceSpec.ROLENAME, roleName);
    }
    
    /**
     * Gets the role name.
     *
     * @return	roleName
     * @see	<code>setRoleName</code>
     */
    public String getRoleName() {
        return spec.getDetail(DataSourceSpec.ROLENAME);
    }

    
    /**
     * Sets the login timeout.
     *
     * @param	loginTimeOut	<code>String</code>
     * @see	<code>getLoginTimeOut</code>
     */
    public void setloginTimeOut(String loginTimeOut) {
        spec.setDetail(DataSourceSpec.LOGINTIMEOUT, loginTimeOut);
    }
    
    /**
     * Gets the login timeout.
     *
     * @return	loginTimeout
     * @see	<code>setLoginTimeOut</code>
     */
    public String getloginTimeOut() {
        return spec.getDetail(DataSourceSpec.LOGINTIMEOUT);
    }
    
    /**
     * Sets the login timeout.
     *
     * @param	loginTimeOut	<code>String</code>
     * @see	<code>getLoginTimeOut</code>
     */
    public void setLoginTimeOut(String loginTimeOut) {
        spec.setDetail(DataSourceSpec.LOGINTIMEOUT, loginTimeOut);
    }
    
    /**
     * Gets the login timeout.
     *
     * @return	loginTimeout
     * @see	<code>setLoginTimeOut</code>
     */
    public String getLoginTimeOut() {
        return spec.getDetail(DataSourceSpec.LOGINTIMEOUT);
    }
    
    /**
     * Sets the delimiter.
     *
     * @param	delim	<code>String</code>
     * @see	<code>getDelimiter</code>
     */
    public void setdelimiter(String delim) {
        spec.setDetail(DataSourceSpec.DELIMITER, delim);
    }
    
    /**
     * Gets the delimiter.
     *
     * @return	delim
     * @see	<code>setDelimiter</code>
     */
    public String getdelimiter() {
        return spec.getDetail(DataSourceSpec.DELIMITER);
    }
    
    /**
     * Sets the delimiter.
     *
     * @param	delim	<code>String</code>
     * @see	<code>getDelimiter</code>
     */
    public void setDelimiter(String delim) {
        spec.setDetail(DataSourceSpec.DELIMITER, delim);
    }
    
    /**
     * Gets the delimiter.
     *
     * @return	delim
     * @see	<code>setDelimiter</code>
     */
    public String getDelimiter() {
        return spec.getDetail(DataSourceSpec.DELIMITER);
    }
    
    /**
     * Sets the driver specific properties.
     *
     * @param	driverProps	<code>String</code>
     * @see	<code>getDriverProperties</code>
     */
    public void setdriverProperties(String driverProps) {
        spec.setDetail(DataSourceSpec.DRIVERPROPERTIES, driverProps);
    }
    
    /**
     * Gets the driver specific properties.
     *
     * @return	driverProps
     * @see	<code>setDriverProperties</code>
     */
    public String getdriverProperties() {
        return spec.getDetail(DataSourceSpec.DRIVERPROPERTIES);
    }
    
    /**
     * Sets the driver specific properties.
     *
     * @param	driverProps	<code>String</code>
     * @see	<code>getDriverProperties</code>
     */
    public void setDriverProperties(String driverProps) {
        spec.setDetail(DataSourceSpec.DRIVERPROPERTIES, driverProps);
    }
    
    /**
     * Gets the driver specific properties.
     *
     * @return	driverProps
     * @see	<code>setDriverProperties</code>
     */
    public String getDriverProperties() {
        return spec.getDetail(DataSourceSpec.DRIVERPROPERTIES);
    }

    protected ManagedConnection constructManagedConnection(PooledConnection pc,
        Connection sqlCon, PasswordCredential passCred, 
        ManagedConnectionFactory mcf ) throws ResourceException 
    {
        
        com.sun.gjc.spi.ManagedConnection mc = 
            new com.sun.gjc.spi.ManagedConnection( pc, sqlCon, passCred, mcf );
        
        return mc;

    }    

}
