/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.jbi.serviceengine.work;

import java.util.logging.Logger;
import javax.jbi.messaging.MessageExchange;
import javax.jbi.messaging.DeliveryChannel;
import com.sun.enterprise.jbi.serviceengine.ServiceEngineException;
import com.sun.corba.ee.spi.orbutil.threadpool.Work;
import com.sun.enterprise.jbi.serviceengine.core.JavaEEServiceEngineContext;
import com.sun.logging.LogDomains;

/**
 * Represents one piece of work that will be submitted to the workqueue.
 *
 * @author Binod P.G	
 */
public abstract class OneWork implements Work {

    private long nqTime;
    protected static final Logger logger =
        LogDomains.getLogger(LogDomains.SERVER_LOGGER);
    private MessageExchange me = null;
    private DeliveryChannel channel = null;
    private boolean useCurrentThread = true;
    private JavaEEServiceEngineContext seContext; 
    private WorkManager wm = null;
    private ServiceEngineException exception = null;


    /**
     * Initializes the work. Save a local copy of delivery channel
     * and work manager.
     */
    public OneWork() {
        this.channel = JavaEEServiceEngineContext.getInstance().getDeliveryChannel();
        this.wm = JavaEEServiceEngineContext.getInstance().getWorkManager();
    }

    /**
     * This method is executed by thread pool as the basic work operation.
     */
    public abstract void doWork(); 
    
    /**
     * Time at which this work is enqueued.
     *
     * @param tme Time in milliseconds.
     */
    public void setEnqueueTime(long tme) {
        this.nqTime = tme;
    }

    /**
     * Retrieves the time at which this work is enqueued
     *
     * @return Time in milliseconds.
     */
    public long getEnqueueTime() {
        return nqTime;
    }

    /**
     * Set the MEP associated with this piece of work.
     */
    public void setMessageExchange(MessageExchange me) {
        this.me = me;
    }

    /**
     * Retrieves the MEP.
     */
    public MessageExchange getMessageExchange() {
        return me;
    }

    /**
     * Set a boolean indicating whether current thread should be used
     * for execution of this work.
     *
     * @param flag If set to true, then the current callng thread will
     * be used for executing the thread. If set to false, the work will
     * be submitted to the Queue of the thread pool.
     */
    public void setUseCurrentThread(boolean flag) {
        this.useCurrentThread = flag;
    }

    /**
     * Retrieves the flag indicating whether the current thread should 
     * be used for work execution or not.
     */
    public boolean getUseCurrentThread() {
        return this.useCurrentThread;
    }

    /**
     * Retrieves the work manager instance.
     */
    public WorkManager getWorkManager() {
        return this.wm;
    }

    /**
     * Retrieves the delivery channel object.
     */
    public DeliveryChannel getDeliveryChannel() {
        return this.channel;
    }

    /**
     * Get the exception, if any produced while executng
     * this work.
     */
    public ServiceEngineException getException() {
        return this.exception;
    }

    /**
     * Convenience method to set the execption object 
     * produced while executing the work.
     */
    public void setException(Throwable t) {
        this.exception = new ServiceEngineException(t);
    }

    /**
     * Execute the work. If current thread should be used,
     * doWork is called directly. Otherwise, work is submitted
     * to the thread pool.
     */
    protected void execute() {
        if (getUseCurrentThread()) {
            doWork();
        } else {
            getWorkManager().submitWork(this);
        }
    }

    /**
     * Retrieves the name of the work.
     *
     * @return Name of the work.
     */
    public String getName() {
        return "One JBI Work";
    }
}
