/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.deployment;

import java.util.Set;

import com.sun.enterprise.deployment.types.MessageDestinationReference;
import com.sun.enterprise.util.LocalStringManagerImpl;


/**
 * An object representing the use of a message destination in a 
 * J2EE component.
 *
 * @author Kenneth Saks
 *
*/

public class MessageDestinationReferenceDescriptor extends EnvironmentProperty 
    implements MessageDestinationReference {

    // Usage types
    public static final String CONSUMES = "Consumes";
    public static final String PRODUCES = "Produces";
    public static final String CONSUMES_PRODUCES = "ConsumesProduces";
    
    private static LocalStringManagerImpl localStrings =
        new LocalStringManagerImpl(MessageDestinationReferenceDescriptor.class);
    
    private BundleDescriptor referringBundle;

    private String usage;

    private String destinationType;

    // JNDI name of physical destination
    private String jndiName;

    // Holds information about the destination to which we are linked.
    private MessageDestinationReferencerImpl referencer;
    
    /** 
    * copy constructor
    */
    public MessageDestinationReferenceDescriptor(MessageDestinationReferenceDescriptor other) {
	super(other);
	referringBundle = other.referringBundle; // copy as-is
	usage = other.usage; // immutable String
	destinationType = other.destinationType; // immutable String
        referencer = new MessageDestinationReferencerImpl(other.referencer);
    }

    /** 
     * Construct an message destination reference 
     * with the given name and descriptor of the reference.
     *
     * @param name the message-destination-ref name as used in 
     * the referencing component
     * @param optional description
     */
    public MessageDestinationReferenceDescriptor(String name, String desc) {
	super(name, "", desc);
        referencer = new MessageDestinationReferencerImpl(this);
    }
    
    /** 
    * Constructs a reference in the extrernal state.
    */
    
    public MessageDestinationReferenceDescriptor() {
        referencer = new MessageDestinationReferencerImpl(this);
    }
    
    /**
     * @return the usage type of the message destination reference
     * (Consumes, Produces, ConsumesProduces)
     */
    public String getUsage() {
        return usage;
    }

    /**
     * @param usage the usage type of the message destination reference
     * (Consumes, Produces, ConsumesProduces)
     */
    public void setUsage(String destUsage) {
        usage = destUsage;
    }

    public String getDestinationType() {
        return destinationType;
    }

    public void setDestinationType(String type) {
        destinationType = type;
    }

    public String getJndiName() {
        return (jndiName != null  && ! jndiName.equals("")) ? 
            jndiName : mappedName;
    }

    public void setJndiName(String physicalDestinationName) {
        jndiName = physicalDestinationName;
    }

    public String getInjectResourceType() {
        return getDestinationType();
    }

    public void setInjectResourceType(String resourceType) {
        setDestinationType(resourceType);
    }

    /**
     * Set the referring bundle, i.e. the bundle within which this
     * message destination reference is declared. 
     */
    public void setReferringBundleDescriptor(BundleDescriptor referringBundle)
    {
	this.referringBundle = referringBundle;
    }

    /**
     * Get the referring bundle, i.e. the bundle within which this
     * message destination reference is declared.  
     */
    public BundleDescriptor getReferringBundleDescriptor()
    {
	return referringBundle;
    }

    //
    // Implementations of MessageDestinationReferencer methods.
    //

    public boolean isLinkedToMessageDestination() {
        return referencer.isLinkedToMessageDestination();
    }
    
    /** 
     * @return the name of the message destination to which I refer 
     */
    public String getMessageDestinationLinkName() {
        return referencer.getMessageDestinationLinkName();
    }

    /** 
     * Sets the name of the message destination to which I refer.
     */
    public void setMessageDestinationLinkName(String linkName) {
        referencer.setMessageDestinationLinkName(linkName);
    }    

    public MessageDestinationDescriptor setMessageDestinationLinkName
        (String linkName, boolean resolveLink) {
        return referencer.setMessageDestinationLinkName(linkName, resolveLink);
    }

    public MessageDestinationDescriptor resolveLinkName() {
        return referencer.resolveLinkName();
    }
        
    public boolean ownedByMessageDestinationRef() {
        return true;
    }

    /**
     * Get the descriptor for the message destination reference owner.
     */ 
    public MessageDestinationReferenceDescriptor getMessageDestinationRefOwner
        () {
        return this;
    }

    /**
     * True if the owner is a message-driven bean.
     */ 
    public boolean ownedByMessageBean() {
        return false;
    }

    /**
     * Get the descriptor for the message-driven bean owner.
     */ 
    public EjbMessageBeanDescriptor getMessageBeanOwner() {
        return null;
    }

    /** 
     * @return the message destination to which I refer. Can be NULL.
    */
    public MessageDestinationDescriptor getMessageDestination() {
        return referencer.getMessageDestination();
    }  

    /**
     * @param messageDestiation the message destination to which I refer.
     */
    public void setMessageDestination(MessageDestinationDescriptor newMsgDest) {
        referencer.setMessageDestination(newMsgDest);
    }

    /** returns a formatted string representing me.
    */
    
    public void print(StringBuffer toStringBuffer) {
        if (isLinkedToMessageDestination()) {
	    toStringBuffer.append("Resolved Message-Destination-Ref ").append(getName()).append( 
                "points to logical message destination ").append(getMessageDestination().getName());
	} else {
	    toStringBuffer.append("Unresolved Message-Destination-Ref ").append(getName()).append(
                "@").append(getType()).append("@").append(usage);
	}	
    }
    
    /* Equality on name. */
    public boolean equals(Object object) {
	if (object instanceof MessageDestinationReference) {
	    MessageDestinationReference reference = 
                (MessageDestinationReference) object;
	    return reference.getName().equals(this.getName());
	}
	return false;
    }
}
