/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.deployment.node.ejb;

import java.util.Map;
import java.util.logging.Level;
import org.w3c.dom.Node;

import com.sun.enterprise.deployment.node.XMLElement;
import com.sun.enterprise.deployment.node.DeploymentDescriptorNode;
import com.sun.enterprise.deployment.node.MethodNode;

import com.sun.enterprise.deployment.Descriptor;
import com.sun.enterprise.deployment.QueryDescriptor;
import com.sun.enterprise.deployment.MethodDescriptor;
import com.sun.enterprise.deployment.xml.EjbTagNames;
import com.sun.enterprise.deployment.util.DOLUtils;

/**
 * This class is responsible for hanlding the query element
 *
 * @author  Jerome Dochez
 * @version 
 */
public class QueryNode extends DeploymentDescriptorNode {

    private QueryDescriptor descriptor = null;
    
    /** Creates new QueryNode */
    public QueryNode() {
        super();
        registerElementHandler(new XMLElement(EjbTagNames.QUERY_METHOD), 
                                                                MethodNode.class, "setQueryMethodDescriptor");                 
    }

    /**
     * @return the Descriptor subclass that was populated  by reading
     * the source XML file
     */
    public Object getDescriptor() {
        if (descriptor == null) {
            descriptor = (QueryDescriptor) super.getDescriptor();
        } 
        return descriptor;        
    }    
        
    /**
     * all sub-implementation of this class can use a dispatch table to map xml element to
     * method name on the descriptor class for setting the element value. 
     *  
     * @return the map with the element name as a key, the setter method as a value
     */    
    protected Map getDispatchTable() {
        // no need to be synchronized for now
        Map table = super.getDispatchTable();
        table.put(EjbTagNames.EJB_QL, "setQuery");    
        return table;
    }
    
    
    /**
     * receives notiification of the value for a particular tag
     * 
     * @param element the xml element
     * @param value it's associated value
     */
    public void setElementValue(XMLElement element, String value) {    
        if (EjbTagNames.QUERY_RESULT_TYPE_MAPPING.equals(element.getQName())) {
            if (EjbTagNames.QUERY_REMOTE_TYPE_MAPPING.equals(value)) {
                descriptor.setHasRemoteReturnTypeMapping();
            } else if (EjbTagNames.QUERY_LOCAL_TYPE_MAPPING.equals(value)) {            
                descriptor.setHasLocalReturnTypeMapping();
            } else {
                DOLUtils.getDefaultLogger().log(Level.SEVERE, "enterprise.deployment.backend.addDescriptorFailure",
                                new Object[] {((Descriptor) getParentNode().getDescriptor()).getName() , value});
            }
        } else {
            super.setElementValue(element, value);
        }
    }
    
    /**
     * write the descriptor class to a DOM tree and return it
     *
     * @param parent node in the DOM tree 
     * @param node name for the root element of this xml fragment      
     * @param the descriptor to write
     * @return the DOM tree top node
     */
    public Node writeDescriptor(Node parent, String nodeName, QueryDescriptor descriptor) {        
        Node queryNode = super.writeDescriptor(parent, nodeName, descriptor);

        writeLocalizedDescriptions(queryNode, descriptor);
                
        // query-method
        MethodNode methodNode = new MethodNode();
        methodNode.writeQueryMethodDescriptor(queryNode, EjbTagNames.QUERY_METHOD, 
                                                                         descriptor.getQueryMethodDescriptor());
        
        if (descriptor.getHasRemoteReturnTypeMapping()) {            
            appendTextChild(queryNode, EjbTagNames.QUERY_RESULT_TYPE_MAPPING, 
                                                    EjbTagNames.QUERY_REMOTE_TYPE_MAPPING);     
        } else {
	    if (descriptor.getHasLocalReturnTypeMapping()) {
                appendTextChild(queryNode, EjbTagNames.QUERY_RESULT_TYPE_MAPPING,
                                                    EjbTagNames.QUERY_LOCAL_TYPE_MAPPING);
            }
	}
        // ejbql element is mandatory.  If no EJB QL query has been
        // specified for the method, the xml element will be empty
        String ejbqlText = descriptor.getIsEjbQl() ? descriptor.getQuery() : "";        
        Node child = appendChild(queryNode, EjbTagNames.EJB_QL);
        child.appendChild(getOwnerDocument(child).createTextNode(ejbqlText));          
        
        return queryNode;
    }
}
