/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.ejb.base.stats;

import javax.management.j2ee.statistics.CountStatistic;
import javax.management.j2ee.statistics.RangeStatistic;
import javax.management.j2ee.statistics.TimeStatistic;

import com.sun.ejb.spi.stats.MonitorableSFSBStoreManager;

import com.sun.enterprise.admin.monitor.stats.AverageRangeStatistic;
import com.sun.enterprise.admin.monitor.stats.BoundedRangeStatisticImpl;
import com.sun.enterprise.admin.monitor.stats.CountStatisticImpl;
import com.sun.enterprise.admin.monitor.stats.MutableAverageRangeStatisticImpl;
import com.sun.enterprise.admin.monitor.stats.MutableBoundedRangeStatisticImpl;
import com.sun.enterprise.admin.monitor.stats.MutableCountStatisticImpl;
import com.sun.enterprise.admin.monitor.stats.MutableTimeStatisticImpl;
import com.sun.enterprise.admin.monitor.stats.TimeStatisticImpl;

/**
 * Implementation of StatefulSessionStoreStats
 *  There is once instance of this class per StatefulEJBContainer
 *
 * @author Mahesh Kannan
 */

public class HAStatefulSessionStoreStatsImpl
    extends StatefulSessionStoreStatsImpl
    implements com.sun.enterprise.admin.monitor.stats.HAStatefulSessionStoreStats
{

    private MutableCountStatisticImpl		checkpointCount;
    private MutableCountStatisticImpl		checkpointSuccessCount;
    private MutableCountStatisticImpl		checkpointErrorCount;
    private MutableAverageRangeStatisticImpl	checkpointSize;
    private MutableAverageRangeStatisticImpl	checkpointTime;

    private Object checkpointCountLock = new Object();
    private Object checkpointSizeLock = new Object();
    private Object checkpointTimeLock = new Object();

    private long    checkpointCountVal;
    private long    checkpointSuccessCountVal;
    private long    checkpointErrorCountVal;

    public HAStatefulSessionStoreStatsImpl(
	MonitorableSFSBStoreManager provider)
    {
	super(provider, "com.sun.enterprise.admin.monitor.stats.HAStatefulSessionStoreStats");
	initialize();
    }

    protected void initialize() {
	super.initialize();

	synchronized (checkpointCountLock) {
	    checkpointCount = new MutableCountStatisticImpl(
		new CountStatisticImpl("CheckpointCount"));
	    checkpointSuccessCount = new MutableCountStatisticImpl(
		new CountStatisticImpl("CheckpointSuccessCount"));
	    checkpointErrorCount = new MutableCountStatisticImpl(
		new CountStatisticImpl("CheckpointErrorCount"));
	}

	long now = System.currentTimeMillis();
	synchronized (checkpointTimeLock) {
	    checkpointTime = new MutableAverageRangeStatisticImpl(
	    	new BoundedRangeStatisticImpl(0, 0, 0,
                                     0, 0, "CheckpointTime",
                                     "millis", "Time spent on checkpointing", 0, 0)
	    );
	}

	synchronized (checkpointSizeLock) {
	    checkpointSize = new MutableAverageRangeStatisticImpl(
	    	new BoundedRangeStatisticImpl(0, 0, 0,
                                     0, 0, "CheckpointSize",
                                     "millis", "Number of bytes checkpointed", 0, 0)
	    );
	}
    }

    /**
     * Returns the total number of sessions checkpointed into the store
     */
    public CountStatistic getCheckpointCount() {
	synchronized (checkpointCountLock) {
	    checkpointCount.setCount(checkpointCountVal);
	   return (CountStatistic) checkpointCount.unmodifiableView();
	}
    }

    /**
     * Returns the total number of sessions successfully Checkpointed into the store
     */
    public CountStatistic getCheckpointSuccessCount() {
	synchronized (checkpointCountLock) {
	    checkpointSuccessCount.setCount(checkpointSuccessCountVal);
	    return (CountStatistic) checkpointSuccessCount.unmodifiableView();
	}
    }

    /**
     * Returns the total number of sessions that couldn't be Checkpointed into the store
     */
    public CountStatistic getCheckpointErrorCount() {
	synchronized (checkpointCountLock) {
	    checkpointErrorCount.setCount(checkpointErrorCountVal);
	    return (CountStatistic) checkpointErrorCount.unmodifiableView();
	}
    }

    /**
     * Returns the number of bytes checkpointed
     */
    public AverageRangeStatistic getCheckpointedBeanSize() {
	synchronized (checkpointTimeLock) {
	    return (AverageRangeStatistic) checkpointSize.unmodifiableView();
	}
    }

    /**
     * Returns the time spent on passivating beans to the store including total, min, max
     */
    public AverageRangeStatistic getCheckpointTime() {
	synchronized (checkpointTimeLock) {
	    return (AverageRangeStatistic) checkpointTime.unmodifiableView();
	}
    }

    public void incrementCheckpointCount(boolean success) {
	synchronized (checkpointCountLock) {
	    checkpointCountVal++;
	    if (success) {
		checkpointSuccessCountVal++;
	    } else {
		checkpointErrorCountVal++;
	    }
	}     
    }

    public void setCheckpointSize(long val) {
	synchronized (checkpointSizeLock) {
	    checkpointSize.setCount(val);
	}
    }

    public void setCheckpointTime(long val) {
	synchronized (checkpointTimeLock) {
	    checkpointTime.setCount(val);
	}
    }

    protected void appendStats(StringBuffer sbuf) {
	super.appendStats(sbuf);
	sbuf.append("CheckpointCount: ").append(checkpointCountVal)
	    .append("; ")
	    .append("CheckpointSuccessCount: ").append(checkpointSuccessCountVal)
	    .append("; ")
	    .append("CheckpointErrorCount: ").append(checkpointErrorCountVal)
	    .append("; ");

	appendTimeStatistic(sbuf, "CheckpointTime", checkpointTime);
    }

}
