/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.iiop;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.LinkedList;
import java.util.logging.Logger;
import java.util.logging.Level;

import com.sun.logging.LogDomains;

import org.omg.CORBA.Any;
import org.omg.IOP.Codec;
import org.omg.IOP.CodecFactory;
import org.omg.IOP.CodecFactoryHelper;
import org.omg.IOP.Encoding;
import org.omg.IOP.ENCODING_CDR_ENCAPS;
import org.omg.IOP.TaggedComponent;
import org.omg.PortableInterceptor.IORInfo;
import org.omg.PortableInterceptor.IORInterceptor;
import org.omg.PortableInterceptor.ObjectReferenceTemplate;
import org.omg.PortableInterceptor.ORBInitializer;
import org.omg.PortableInterceptor.ORBInitInfo;
import org.omg.PortableInterceptor.ORBInitInfoPackage.DuplicateName;

// BEGIN imports for cluster info.
import com.sun.enterprise.admin.util.JMXConnectorConfig;
import com.sun.enterprise.config.serverbeans.IiopListener;
import com.sun.enterprise.config.ConfigException;
import com.sun.enterprise.config.ConfigContext;
import com.sun.enterprise.server.ApplicationServer;
import com.sun.enterprise.config.serverbeans.Cluster;
import com.sun.enterprise.config.serverbeans.ClusterHelper;
import com.sun.enterprise.config.serverbeans.ServerRef;
import com.sun.enterprise.config.serverbeans.Server;
import com.sun.enterprise.config.serverbeans.Config;
import com.sun.enterprise.config.serverbeans.ConfigAPIHelper;
import com.sun.enterprise.config.serverbeans.ServerHelper;
import com.sun.enterprise.config.serverbeans.IiopService;
import com.sun.enterprise.config.serverbeans.PropertyResolver;
// END imports for cluster info.

import com.sun.enterprise.iiop.AlternateIIOPAddressComponent;
import com.sun.enterprise.iiop.AlternateIIOPAddressComponentHelper;

import com.sun.corba.ee.spi.legacy.interceptor.ORBInitInfoExt ;
import com.sun.corba.ee.spi.orb.ORB;

import com.sun.corba.ee.spi.folb.ClusterInstanceInfo;
import com.sun.corba.ee.spi.folb.SocketInfo;

/**
 * @author Harold Carr
 */
public class FailoverIORInterceptor
    extends org.omg.CORBA.LocalObject
    implements ORBInitializer, IORInterceptor
{
    private static Logger _logger = null;
    static {
       _logger = LogDomains.getLogger(LogDomains.CORBA_LOGGER);
    }

    private ORB orb ;

    private static final String baseMsg
	= FailoverIORInterceptor.class.getName();
    private static final String ORB_LISTENTER = "orb-listener";

    private List currentAddressList;
    private List previousAddressList;
    private List randomizedAddressList;
    private List randomizedAndMarshaledAddressList;

    //
    // Interceptor
    //

    public String name()    { return baseMsg; }
    public void   destroy() { }

    //
    // ORBInitializer
    //

    public FailoverIORInterceptor()
    {
    }

    public void pre_init(ORBInitInfo info) { }

    public void post_init(ORBInitInfo info)
    {
	ORB orb = ((ORBInitInfoExt)info).getORB() ;
	try {
	    info.add_ior_interceptor(new FailoverIORInterceptor(orb));
	    _logger.log(Level.FINE, baseMsg + " registered.");
	} catch (DuplicateName ex) {
            _logger.log(Level.WARNING, "DuplicateName from " + baseMsg , ex);
	    /*
	    RuntimeException rte = 
		new RuntimeException("DuplicateName from " + baseMsg , ex);
	    rte.initCause(ex);
	    throw rte;
	    */
	}
    }

    //
    // IORInterceptor
    //

    public FailoverIORInterceptor( ORB orb ) 
    {
	this.orb = orb ;
    }

    public void   establish_components(IORInfo iorInfo)
    {
	try {
	    _logger.log(Level.FINE,
			baseMsg + ".establish_components->:");

	    if (ApplicationServer.getServerContext() == null) {
		_logger.log(Level.FINE,
			    baseMsg + ".establish_components:"
			    + " not adding AlternateIIOPAddressComponent"
			    + " because: no server context");
		return;
	    }

	    ConfigContext configCtx =
		ApplicationServer.getServerContext().getConfigContext();
	    String instanceName =
		ApplicationServer.getServerContext().getInstanceName();
	    Cluster cluster = null;
	    try {
		cluster = 
		    ClusterHelper
		        .getClusterForInstance(configCtx, instanceName);
	    } catch (ConfigException e) {
		_logger.log(Level.FINE, 
			    baseMsg + ".establish_components:" 
			    + " not adding AlternateIIOPAddressComponent"
			    + " because: "
			    + e);
		return;
	    }
	    ServerRef[] servers = cluster.getServerRef();
	    IiopListener[][] listeners = getIIOPEndPointsForCurrentCluster(
                configCtx, instanceName, servers);
	    currentAddressList = new ArrayList();
	    for (int i = 0; i < servers.length; i++) {
		String serverName = servers[i].getRef();
		String hostName =
		    getHostNameForServerInstance(configCtx, serverName);
		if (hostName==null) {
		    hostName = listeners[i][0].getAddress();
		}
		for (int j = 0; j < listeners[i].length; j++) {
		    String id = listeners[i][j].getId();
		    String port = 
			getResolvedPort(configCtx,listeners[i][j], serverName);
		    // REVISIT: Make question an official API.
		    if (! id.startsWith(ORB_LISTENTER)) {
			if (_logger.isLoggable(Level.FINE)) {
			    _logger.log(Level.FINE, 
					baseMsg + ".establish_components:"
					+ " ignoring SSL ports: "
					+ id + " " + hostName + " " + port);
			}
			continue;
		    }

		    if (_logger.isLoggable(Level.FINE)) {
			_logger.log(Level.FINE, 
				    baseMsg + ".establish_components:"
				    + " adding AlternateIIOPAddressComponent"
				    + " for listener id: " 
				    + id
				    + " address/port: "
				    + hostName
				    + "/"
				    + port);
		    }

		    AlternateIIOPAddressComponent address =
			new AlternateIIOPAddressComponent(
                            hostName,
			    Integer.decode(port).shortValue());

		    currentAddressList.add(address);
		}
	    }

	    if (previousAddressList == null) {

		//
		// Save the previous list for subsequent comparisons.
		//

		previousAddressList = new ArrayList();
		// List does not define .clone and Object.clone is protected.
		Iterator i = currentAddressList.iterator();
		while (i.hasNext()) { previousAddressList.add(i.next()); }

		//
		// Randomize the list for this ORB.
		// Save it in unmarshaled form for logging.
		//

		java.util.Collections.shuffle(currentAddressList);
		randomizedAddressList = currentAddressList;

		if (_logger.isLoggable(Level.FINE)) {
		    _logger.log(Level.FINE, baseMsg 
				+ " first call - saving randomized alternate address list: "
				+ randomizedAddressList);
		}

		//
		// Save a marshaled version for subsequent calls.
		//

		randomizedAndMarshaledAddressList = new ArrayList();
		i = randomizedAddressList.iterator();
		while (i.hasNext()) {
		    AlternateIIOPAddressComponent address =
			(AlternateIIOPAddressComponent) i.next();
		    Any any = orb.create_any();
		    AlternateIIOPAddressComponentHelper.insert(any, address);
		    byte[] data = getCodec(orb).encode_value(any);
		    TaggedComponent tc =
			new TaggedComponent(org.omg.IOP.TAG_ALTERNATE_IIOP_ADDRESS.value,
					    //AlternateIIOPAddressComponent.TAG_ALTERNATE_IIOP_ADDRESS_ID,
					    data);
		    randomizedAndMarshaledAddressList.add(tc);
		}
	    } else {
		if (! currentAddressList.equals(previousAddressList)) {
		    throw new RuntimeException(
		        "Current address list: "
			+ currentAddressList
			+ " different from previous list: "
			+ previousAddressList);
		}
	    }

	    Iterator i = randomizedAndMarshaledAddressList.iterator();
	    while (i.hasNext()) {
		iorInfo.add_ior_component((TaggedComponent) i.next());
	    }

	    if (_logger.isLoggable(Level.FINE)) {
		_logger.log(Level.FINE, baseMsg 
			    + " adding randomized alternate addresses: "
			    + randomizedAddressList);
	    }
	} catch (Throwable e) {
            _logger.log(Level.WARNING, 
			"Problem in " + baseMsg + ".establish_components", 
			e);
	    /*
	    RuntimeException rte = 
		new RuntimeException("Problem in " + baseMsg + ".establish_components");
	    rte.initCause(e);
	    throw rte;
	    */
	} finally {
	    _logger.log(Level.FINE, baseMsg + ".establish_components<-:");
	}
    }

    public void components_established( IORInfo iorInfo )
    {
    }

    public void adapter_manager_state_changed( int managerId, short state )
    {
    }

    public void adapter_state_changed( ObjectReferenceTemplate[] templates,
				       short state ) 
    {
    }

    ////////////////////////////////////////////////////
    //
    // Implementation.
    //

    public static List<ClusterInstanceInfo> getClusterInstanceInfo()
    {
	List<ClusterInstanceInfo> result = 
	    new LinkedList<ClusterInstanceInfo>();

	try {
	    _logger.log(Level.FINE,
			baseMsg + ".getClusterForInstanceInfo->:");

	    if (ApplicationServer.getServerContext() == null) {
		_logger.log(Level.FINE,
			    baseMsg + ".getClusterForInstanceInfo:"
			    + " not adding addresses "
			    + " because: no server context");
		return null;
	    }

	    ConfigContext configCtx =
		ApplicationServer.getServerContext().getConfigContext();
	    String instanceName =
		ApplicationServer.getServerContext().getInstanceName();
	    Cluster cluster = null;
	    try {
		cluster = 
		    ClusterHelper
		        .getClusterForInstance(configCtx, instanceName);
	    } catch (ConfigException e) {
		_logger.log(Level.FINE, 
			    baseMsg + ".getClusterForInstance: " 
			    + e);
		return null;
	    }
	    ServerRef[] servers = cluster.getServerRef();
	    IiopListener[][] listeners = getIIOPEndPointsForCurrentCluster(
                configCtx, instanceName, servers);
	    for (int i = 0; i < servers.length; i++) {
		String serverName = servers[i].getRef();
		String hostName =
		    getHostNameForServerInstance(configCtx, serverName);
		if (hostName==null) {
		    hostName = listeners[i][0].getAddress();
		}
		List<SocketInfo> listOfSocketInfo =
		    new LinkedList<SocketInfo>();
		for (int j = 0; j < listeners[i].length; j++) {
		    String id = listeners[i][j].getId();
		    String port = 
			getResolvedPort(configCtx,listeners[i][j], serverName);
		    if (_logger.isLoggable(Level.FINE)) {
			_logger.log(Level.FINE, 
				    baseMsg + ".getClusterForInstance:"
				    + " adding address for "
				    + serverName + "/" + id 
				    + "/" + hostName + "/" + port);
		    }
		    listOfSocketInfo.add(
		        new SocketInfo(id, hostName, Integer.valueOf(port)));
		}
		// REVISIT - make orbutil utility
		// and use in IiopFolbGmsClient and here.
		SocketInfo[] arrayOfSocketInfo =
		    new SocketInfo[listOfSocketInfo.size()];
		int x = 0;
		for (SocketInfo si : listOfSocketInfo) {
		    arrayOfSocketInfo[x++] = si;
		}
		// REVISIT - default 100 weight
		// Not used by JNLP but should put in the actual weight.
		ClusterInstanceInfo clusterInstanceInfo = 
		    new ClusterInstanceInfo(serverName, 100, arrayOfSocketInfo);
		result.add(clusterInstanceInfo);
		if (_logger.isLoggable(Level.FINE)) {
		    _logger.log(Level.FINE, 
				baseMsg + ".getClusterForInstance: " +
				ASORBUtilities.toString(clusterInstanceInfo));
		}
	    }
	    return result;
	} catch (ConfigException e) {
	    return null;
	} finally {
	    _logger.log(Level.FINE,
			baseMsg + ".getClusterForInstanceInfo<-: " + result);
	}
    }

    /**
      * The following returns the IIOP listener(s) for the all
      * server belonging to the current cluster.
      *
      * @author  satish.viswanatham@sun.com
      *
      * @param   configCtx             Current server's config context
      * @param   instanceName          instance name of the current server
      *
      * @return  IiopListener[i][j]    Two dimension array of iiop listener info
      *                                i = index of server 
      *                                j = index of iiop listner (j <= 3)
      * @throws  ConfigException       In any configuration error cases, 
      *                                ConfigException is thrown
      */
    private static IiopListener[][] getIIOPEndPointsForCurrentCluster(
        ConfigContext configCtx, String instanceName, ServerRef[] servers)
	throws ConfigException
    {
	// For each server instance in a cluster, there are 3 iiop listeners:
	// one for non ssl, one for ssl and third for ssl mutual auth
	
	IiopListener[][] listeners = new IiopListener[servers.length][3];
	for (int i = 0; i < servers.length; i++) {
	    Server server = 
		ServerHelper.getServerByName(configCtx, servers[i].getRef());
	    String configRef = server.getConfigRef();
	    Config config =
		ConfigAPIHelper.getConfigByName(configCtx, configRef);
	    IiopService iiopService = config.getIiopService();
	    listeners[i] = iiopService.getIiopListener();
	}
	return listeners;
    }

    /**
     * Returns ip address from node agent refered from instance
     * or null if Exception
     *
     * @author  sridatta.viswanath@sun.com
     */
    private static String getHostNameForServerInstance(ConfigContext ctx, 
						       String serverName) 
    {
        try {
            JMXConnectorConfig info = 
		ServerHelper.getJMXConnectorInfo(ctx, serverName);
            _logger.log(Level.FINE, 
			baseMsg + ".getHostNameForServerInstance: " +
			"found info: " + info.toString());
	    String host = info.getHost();
            _logger.log(Level.FINE, 
			baseMsg + ".getHostNameForServerInstance: " +
			"found host: " + host);
            return host;
        } catch (Throwable e){
            _logger.log(Level.FINE, 
			baseMsg + ".getHostNameForServerInstance: " +
			"gotException: " + e + " " + e.getMessage() +
			"; returning null");
            return null;
        }
    }

    /**
     * Gets the correct resolved value for the specific instance
     * Without this routine, config context resolves the value
     * to the current running instance
     *
     * @author  sridatta.viswanath@sun.com
     */
    private static String getResolvedPort(ConfigContext ctx, 
					  IiopListener l, 
					  String server) 
	throws ConfigException 
    {
	String rawPort = l.getRawAttributeValue("port");
	PropertyResolver pr = new PropertyResolver(ctx, server);
	return pr.resolve(rawPort);
    }

    static Codec getCodec(ORB orb)
    {
	Codec result = null;
	try {
	    CodecFactory codecFactory = 
		CodecFactoryHelper.narrow(
                    orb.resolve_initial_references("CodecFactory"));
	    result = codecFactory.create_codec(
                new Encoding((short)ENCODING_CDR_ENCAPS.value, 
			     (byte)1, (byte)2));
	} catch (Throwable e) {
            _logger.log(Level.WARNING, 
			"Problem in " + baseMsg + ".getCodec",
			e);
	}
	return result;
    }
}

// End of file.
