/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

/*
 * @(#) ReloadMonitor.java
 *
 * Copyright 2000-2001 by iPlanet/Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of iPlanet/Sun Microsystems, Inc. ("Confidential Information").
 * You shall not disclose such Confidential Information and shall
 * use it only in accordance with the terms of the license
 * agreement you entered into with iPlanet/Sun Microsystems.
 */
package com.sun.enterprise.server;

import java.io.File;
import java.util.ArrayList;
import java.util.Iterator;

import java.util.logging.Level;
import java.util.logging.Logger;
import com.sun.logging.LogDomains;
import com.sun.enterprise.admin.server.core.channel.ReconfigHelper;

/**
 * Monitors dynamic reload files for applications and stand alone modules 
 * from the server runtime. 
 *
 * @atuthor Nazrul Islam
 * @since   JDK 1.4
 */
class ReloadMonitor extends AbstractMonitor {

    /** name of the file that user will update time stamp to trigger a reload */
    static final String RELOAD_FILE = ".reload";
    
    static Logger _logger = LogDomains.getLogger(LogDomains.CORE_LOGGER);

    /** singleton instance */
    private static ReloadMonitor _instance = null;

    /**
     * Constructor - prohibits anyone from constructing this object.
     *
     * @param    pollInterval    polling interval
     */
    private ReloadMonitor(long pollInterval) {
        super(pollInterval);
    }

    /**
     * Returns the singleton instance. If the object was not created already, 
     * it uses the given polling interval. 
     *
     * @param    pollInterval    polling interval
     */
    static ReloadMonitor getInstance(long pollInterval) {
        if (_instance == null) {
            _instance = new ReloadMonitor(pollInterval);
        }
        return _instance;
    }

    /** 
     * Removes the given application or stand alone module from the 
     * monitored list.
     *
     * @param    id    registration name of application
     *
     * @return   true if removed successfully
     */
    boolean removeMonitoredEntry(String id) {

        boolean removed = false;

        // returns false if application name is null
        if (id == null) {
            return removed;
        }

        synchronized (this._monitoredEntries) {
            Iterator iter = this._monitoredEntries.iterator();
            while (iter.hasNext()) {
                MonitorableEntry entry = (MonitorableEntry) iter.next();
                if ( id.equals(entry.getId()) ) {
                    this._monitoredEntries.remove(entry);
                    removed = true;
                    break;
                }
            }
        }

        return removed;
    }

    /**
     * This method gets called from the monitor thread. This goes through 
     * all the monitored entried and checks the time stamps. If any of the 
     * time stamps is modified, it makes a callback to its listener. The 
     * callbacks are single threaded, i.e., waits for one to finish before 
     * makes the second call.
     *
     * <p> The time stamp of the monitored entry is set to the current 
     * time stamp before the callback is made. 
     */
    public void run() {

        try {
            ArrayList reloadList = new ArrayList();

            synchronized (_monitoredEntries) {
                Iterator iter = _monitoredEntries.iterator();
                MonitorableEntry entry = null;

                while (iter.hasNext()) {
                    entry                = (MonitorableEntry) iter.next();
                    File file            = entry.getMonitoredFile();
                    long lastModified    = file.lastModified();
                    long lastReloadedAt  = entry.getLastReloadedTimeStamp();

                    // time stamp is updated
                    if (lastModified > lastReloadedAt) {
                        // sets the time stamp so that it gets called once
                        entry.setLastReloadedTimeStamp(lastModified);

                        reloadList.add(entry);
                    }
                }
            }

            // found some entries with modified time stamp
            if (reloadList.size() > 0) {

                _logger.log(Level.FINEST,
                    "[ReloadMonitor] Monitor detected reloadable entry!");

                int size = reloadList.size();
                MonitorableEntry entry = null;

                for (int i=0; i<size; i++) {
                    entry = (MonitorableEntry) reloadList.get(i);

                    MonitorListener l = entry.getListener();

                    // calls back the listener
                    boolean success = l.reload(entry);

                    // log status
                    if (success) {
                        _logger.log(Level.INFO,
                            "core.application_reload_successful",
                            entry.getDisplayName());
                    } else {
                        _logger.log(Level.INFO,
                            "core.application_reload_failed",
                            entry.getDisplayName());
                    }
                }

                // Reload the web modules
                /*
                 * Remove compile time dependence on J2EERunner. Replace it
                 * temporarily with call to ReconfigHelper.
                J2EERunner.requestReconfiguration();
                */
                ReconfigHelper.sendReconfigMessage("");

                // removes all the entries after the call back
                reloadList.clear();
            }

        } catch (Throwable t) { 
            // catches any uncaught exceptions thrown from the handlers
            _logger.log(Level.WARNING, "core.exception", t);
        }
    }
}
