/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.webservice;

import java.lang.UnsupportedOperationException;

import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;

import java.util.logging.Logger;
import java.util.logging.Level;
import com.sun.logging.LogDomains;

/**
 * InvocationHandler used to delegate calls to JAXRPC servlet impls
 * that aren't subtypes of their associated Service Endpoint Interface.
 *
 * @author Kenneth Saks
 */
public class ServletImplInvocationHandler implements InvocationHandler {

    private static Logger logger = LogDomains.getLogger(LogDomains.WEB_LOGGER);

    private Object servletImplDelegate;
    private Class servletImplClass;
    
    public ServletImplInvocationHandler(Object delegate) {
        servletImplDelegate = delegate;
        servletImplClass    = delegate.getClass();
    }
    
    public Object invoke(Object proxy, Method method, Object[] args) 
        throws Throwable {

        // NOTE : be careful with "args" parameter.  It is null
        //        if method signature has 0 arguments.

        Class methodClass = method.getDeclaringClass();
        if( methodClass == java.lang.Object.class )  {
            return invokeJavaObjectMethod(this, method, args);
        }

        Object returnValue = null;

        try {
            // Since impl class isn't subtype of SEI, we need to do a 
            // method lookup to get method object to use for invocation.
            Method implMethod = servletImplClass.getMethod
                (method.getName(), method.getParameterTypes());
            returnValue = implMethod.invoke(servletImplDelegate, args);
        } catch(InvocationTargetException ite) {
            logger.log(Level.FINE, "", ite);
            throw ite.getCause();
        } catch(Throwable t) {
            logger.log(Level.INFO, "Error invoking servlet impl", t);
            throw t;
        }

	return returnValue;
    }

    private Object invokeJavaObjectMethod(InvocationHandler handler, 
                                          Method method, Object[] args) 
        throws Throwable {

        Object returnValue = null;

        // Can only be one of : 
        //     boolean java.lang.Object.equals(Object)
        //     int     java.lang.Object.hashCode()
        //     String  java.lang.Object.toString()
        //
        // Optimize by comparing as few characters as possible.

        switch( method.getName().charAt(0) ) {
            case 'e' :
                Object other = Proxy.isProxyClass(args[0].getClass()) ?
                    Proxy.getInvocationHandler(args[0]) : args[0];
                returnValue = new Boolean(handler.equals(other));
                break;
            case 'h' :
                returnValue = new Integer(handler.hashCode());
                break;
            case 't' :
                returnValue = handler.toString();
                break;
            default :
                throw new Throwable("Object method " + method.getName() + 
                                    "not found");
        }

        return returnValue;
    }

}
