/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

/*
 * TracingSystemHandlerFactory.java
 *
 * Created on August 12, 2004, 4:04 PM
 */

package com.sun.enterprise.webservice.monitoring;

import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.util.Collection;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.logging.Logger;
import java.util.logging.Level;

import java.io.File;
import java.io.BufferedInputStream;
import java.io.FileInputStream;

import java.net.URL;
import java.net.MalformedURLException;

import org.w3c.dom.Document;
import javax.xml.parsers.DocumentBuilderFactory;
import org.xml.sax.InputSource;
import javax.xml.xpath.XPathFactory;
import javax.xml.xpath.XPath;
import javax.xml.namespace.NamespaceContext;

import javax.enterprise.deploy.shared.ModuleType;

import com.sun.logging.LogDomains;
import com.sun.enterprise.deployment.WebServiceEndpoint;

/**
 * This class acts as a factory to create TracingSystemHandler 
 * instances. It also provides an API to register listeners 
 * of SOAP messages.
 *
 * @author Jerome Dochez
 */
public class WebServiceEngineImpl implements WebServiceEngine {
    
    protected Map<String, Endpoint> endpoints = new HashMap<String, Endpoint>();
    protected List<EndpointLifecycleListener> lifecycleListeners = 
            new ArrayList<EndpointLifecycleListener>();
    protected List<AuthenticationListener> authListeners = 
            new ArrayList<AuthenticationListener>();
    protected GlobalMessageListener globalMessageListener = null; 
           

    static WebServiceEngineFactory factory = WebServiceEngineFactory.getInstance();
    static ThreadLocal servletThreadLocal = new ThreadLocal();
    public static Logger sLogger = LogDomains.getLogger(LogDomains.CORE_LOGGER);
    
    
    /** Creates a new instance of TracingSystemHandlerFactory */
    protected WebServiceEngineImpl() {
    }
    
    public static WebServiceEngineImpl getInstance() {
            
        if (factory.getEngine()==null) {
            synchronized(factory) {                
                if (factory.getEngine()==null) {
                    factory.setEngine(new WebServiceEngineImpl());
                    factory.getEngine().addAuthListener(new LogAuthenticationListener());
                }
            }
        }
        return (WebServiceEngineImpl) factory.getEngine();
    }
    
    private EndpointImpl createHandler(WebServiceEndpoint endpointDesc)  {
        
        EndpointImpl newEndpoint = createEndpointInfo(endpointDesc);
        if (newEndpoint==null) {
            return null;
        }
        String key = pathFromURL(newEndpoint.getEndpointSelector());
        endpoints.put(key, newEndpoint); 
        
        // notify listeners
        for (EndpointLifecycleListener listener : lifecycleListeners) {
            listener.endpointAdded(newEndpoint);
        }
        
        return newEndpoint;
    }
    
    public EndpointImpl createHandler(com.sun.xml.rpc.spi.runtime.SystemHandlerDelegate parent, 
        WebServiceEndpoint endpointDesc)  {
        
        EndpointImpl newEndpoint = createHandler(endpointDesc);
        ((JAXRPCEndpointImpl)newEndpoint).setParent(parent);
        return newEndpoint;
    }

    public EndpointImpl createHandler(com.sun.xml.ws.spi.runtime.SystemHandlerDelegate parent, 
        WebServiceEndpoint endpointDesc)  {
        
        EndpointImpl newEndpoint = createHandler(endpointDesc);
        ((JAXWSEndpointImpl)newEndpoint).setParent(parent);
        return newEndpoint;
    }
    
    public Endpoint getEndpoint(String url) {    
        return endpoints.get(pathFromURL(url));
    }
    
    public Iterator<Endpoint> getEndpoints() {
        return endpoints.values().iterator();
    }
    
    public void removeHandler(WebServiceEndpoint endpointDesc) {

        EndpointImpl endpoint = (EndpointImpl) endpointDesc.getExtraAttribute(EndpointImpl.NAME);
        if (endpoint==null) 
            return;
                
        // remove this endpoint from our list of endpoints
        endpoints.remove(pathFromURL(endpoint.getEndpointSelector()));

        // notify listeners
        for (EndpointLifecycleListener listener : lifecycleListeners) {
            listener.endpointRemoved(endpoint);
        }
        
        // forcing the cleaning so we don't have DOL objects staying alive because
        // some of our clients have not released the endpoint instance.
        endpoint.setDescriptor(null);
    }
            
    public void addLifecycleListener(EndpointLifecycleListener listener) {
        lifecycleListeners.add(listener);
    }
    
    public void removeLifecycleListener(EndpointLifecycleListener listener) {
        lifecycleListeners.remove(listener);
    }
    
    public void addAuthListener(AuthenticationListener listener) {
        authListeners.add(listener);
    }
    
    public void removeAuthListener(AuthenticationListener listener) {
        authListeners.remove(listener);
    }
    
    public Collection<AuthenticationListener> getAuthListeners() {
        return authListeners;
    }
    
    public GlobalMessageListener getGlobalMessageListener() {
        return globalMessageListener;
    }
    
    public void setGlobalMessageListener(GlobalMessageListener listener) {
        globalMessageListener = listener;
    }
    
    
    public boolean hasGlobalMessageListener() {
        return globalMessageListener!=null;
    }
                
    private EndpointImpl createEndpointInfo(WebServiceEndpoint endpoint) {
        
        try {   
            File wsdlFile = new File(endpoint.getWebService().getGeneratedWsdlFilePath());
            if (!wsdlFile.exists()) {
                return null;
            }        

            DocumentBuilderFactory dFactory = DocumentBuilderFactory.newInstance();
            InputSource inputSource = new InputSource(new BufferedInputStream(new FileInputStream(wsdlFile)));
            Document wsdlDoc = dFactory.newDocumentBuilder().parse(new BufferedInputStream(new FileInputStream(wsdlFile)));

             
            XPathFactory xPathFactory = XPathFactory.newInstance();
            XPath xPath = xPathFactory.newXPath();
            NamespaceContext context = new NamespaceContextImpl(wsdlDoc);
            xPath.setNamespaceContext(context);

               String xpathExpression = "/:definitions/:service/:port[@name='"+
                    endpoint.getWsdlPort().getLocalPart()+"']/"+
                     endpoint.getSoapAddressPrefix()+":address/@location";
             
            String endpointURL = xPath.evaluate(xpathExpression, inputSource);
            if (endpointURL==null) {
                System.out.println("Cannot get endpoint URL from " + endpoint.getWsdlPort());
            }   
        
            EndpointType endpointType;            
            ModuleType moduleType = endpoint.getWebService().getWebServicesDescriptor().getModuleType();
            if (moduleType.equals(ModuleType.EJB)) {
                endpointType = EndpointType.EJB_ENDPOINT;
            } else {
                endpointType = EndpointType.SERVLET_ENDPOINT;
            }

            EndpointImpl newEndpoint;
            // At this point, we can depend on presence of mapping file to distinguish between JAXRPC and JAXWS
            // service
            if(endpoint.getWebService().getMappingFileUri()==null) {
                newEndpoint = new JAXWSEndpointImpl(endpointURL, endpointType); 
            } else {
                newEndpoint = new JAXRPCEndpointImpl(endpointURL, endpointType); 
            }
            newEndpoint.setDescriptor(endpoint);
            return newEndpoint;
        
        } catch(Exception e) {
            e.printStackTrace();
        }                         
        return null;
    }    
        
    /** 
     * Callback when a web service request entered the web service container
     * before any processing is done. 
     * @return a message ID to trace the request in the subsequent callbacks
     */
    public String preProcessRequest(Endpoint endpoint) {
        
        if (globalMessageListener==null)
            return null;
        
        return globalMessageListener.preProcessRequest(endpoint);
    }
    
    /** 
     * Callback when a web service request is received on 
     * the endpoint.
     * @param messageID returned by preProcessRequest call 
     * @param msgContext the jaxrpc message trace, transport dependent.
     */
    public void processRequest(String messageID, com.sun.xml.rpc.spi.runtime.SOAPMessageContext context, 
            TransportInfo info) {
        
        if (globalMessageListener==null) 
            return;
        
        globalMessageListener.processRequest(messageID, context, info);
    }
    
    /**
     * Callback when a web service response is received on the 
     * endpoint. 
     * @param messageID returned by the preProcessRequest call
     * @param trace jaxrpc message context
     */
    public void processResponse(String messageID, com.sun.xml.rpc.spi.runtime.SOAPMessageContext context) {
        
        if (globalMessageListener==null)
            return;
        
        globalMessageListener.processResponse(messageID, context);
    }
    
    /** 
     * Callback when a 2.0 web service request is received on 
     * the endpoint.
     * @param messageID returned by preProcessRequest call 
     * @param msgContext the jaxrpc message trace, transport dependent.
     */
    public void processRequest(String messageID, com.sun.xml.ws.spi.runtime.SOAPMessageContext context, 
            TransportInfo info) {
        
        if (globalMessageListener==null) 
            return;

        globalMessageListener.processRequest(messageID, context, info);
    }
    
    /**
     * Callback when a 2.0 web service response is received on the 
     * endpoint. 
     * @param messageID returned by the preProcessRequest call
     * @param trace jaxrpc message context
     */
    public void processResponse(String messageID, com.sun.xml.ws.spi.runtime.SOAPMessageContext context) {
        
        if (globalMessageListener==null)
            return;
        globalMessageListener.processResponse(messageID, context);
    }
    
    /**
     * Callback when a web service response has finished being processed
     * by the container and was sent back to the client
     * @param messageID returned by the preProcessRequest call
     */
    public void postProcessResponse(String messageID, TransportInfo info) {
        
        if (globalMessageListener==null)
            return;
        
        globalMessageListener.postProcessResponse(messageID, info);        
    }
    
    public ThreadLocal getThreadLocal() {
        return servletThreadLocal;
    }
    
    private String pathFromURL(String urlString) {
        URL url;
        try {
            url = new URL(urlString);            
        } catch(MalformedURLException e) {
            return null;
        }
        return url.getPath();
    }
}
