/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.webservice.spi;

import java.util.ArrayList;
import java.util.logging.Level;

import javax.xml.ws.spi.ServiceDelegate;

import com.sun.enterprise.webservice.WsUtil;
import com.sun.enterprise.webservice.ServiceCreationListenerImpl;


/**
 * DefaultServiceDelegateFactory is the factory used by our Provider implementation
 * to create a ServiceDelegate instance. This also defines the ability to plug
 * another ServiceDelegateFactory if necessary as well as providing hooks to 
 * register ServiceCreationListener listeners.
 *
 * @author Jerome Dochez
 */
public class DefaultServiceDelegateFactory implements ServiceDelegateFactory {
    
    static ServiceDelegateFactory factory;
    ArrayList<ServiceDelegateCreationListener> listeners = new ArrayList();
    
    /**
     * Resets the current ServiceDelegateFactory instance responsible for 
     * creating ServiceDelegate instances
     * @param ServiceDelegateFactory the instance
     */
    public static void setFactory(ServiceDelegateFactory f) {
        factory = f;
    }
    
    /**
     * @return the current ServiceDelegateFactory instance
     */
    public static ServiceDelegateFactory getFactory() {
        if (factory==null) {
            synchronized(DefaultServiceDelegateFactory.class) {
                if (factory==null) {
                    factory= new DefaultServiceDelegateFactory();
                }                
            }
        }
        return factory;
    }
    
    /** Creates a new instance of DefaultServiceDelegateFactory */
    protected DefaultServiceDelegateFactory() {
        // add our unique listener instance 
        addListener(ServiceCreationListenerImpl.getDefaultListener());
    }
    
    /**
     * Creates a new ServiceDelegate instance using a delegation instance
     * @param the delegate
     * @return the new ServiceDelegate.
     */
    public ServiceDelegate create(ServiceDelegate delegate) {
        ServiceDelegateImpl ourServiceDelegate = new ServiceDelegateImpl(delegate);
        serviceCreated(ourServiceDelegate);
        return ourServiceDelegate;
    }
    
    /**
     * Add a listener for ServiceDelegate creation
     * @param listener implementation
     */
    public void addListener(ServiceDelegateCreationListener listener) {
        listeners.add(listener);
    }
    
    // ServiceDelegate creation notification
    private void serviceCreated(ServiceDelegateImpl delegate) {
        for (ServiceDelegateCreationListener listener : listeners) {
             try {
                listener.serviceCreated(delegate);
             } catch(Throwable t) {
                 WsUtil.getDefaultLogger().severe("Exception " 
                         + t.getMessage() + " in serviceDelegateCreationListener : " + listener);
                 WsUtil.getDefaultLogger().log(Level.FINE, t.getMessage(), t);
             }
        }
    }    
}
