/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
package oracle.toplink.essentials.internal.ejb.cmp3.metadata;

import java.util.Map;
import java.util.Set;
import java.util.List;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.ArrayList;
import java.util.Collection;

import java.lang.reflect.Type;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;

import oracle.toplink.essentials.sessions.Project;
import oracle.toplink.essentials.internal.helper.Helper;
import oracle.toplink.essentials.mappings.CollectionMapping;
import oracle.toplink.essentials.descriptors.ClassDescriptor;
import oracle.toplink.essentials.exceptions.ValidationException;
import oracle.toplink.essentials.internal.security.PrivilegedAccessController;

/**
 * Common helper methods for the annotation and xml processors.
 * 
 * @author Guy Pelletier, Dave McCann
 * @since TopLink EJB 3.0 Reference Implementation
 */
public class MetadataHelper {
    public static final String IS_PROPERTY_METHOD_PREFIX = "is";
    public static final String GET_PROPERTY_METHOD_PREFIX = "get";
    public static final String SET_PROPERTY_METHOD_PREFIX = "set";
    public static final String SET_IS_PROPERTY_METHOD_PREFIX = "setIs";
    private static final int POSITION_AFTER_IS_PREFIX = IS_PROPERTY_METHOD_PREFIX.length();
    private static final int POSITION_AFTER_GET_PREFIX = GET_PROPERTY_METHOD_PREFIX.length();
    
    // Holds the field classifications for temporal types.
    protected static HashMap m_fieldClassifications;
    
    // Holds the disciminator column class types for discriminator types.
    protected static HashMap m_discriminatorTypes;

    /**
     * INTERNAL:
     * Search the given sessions list of ordered descriptors for a descriptor 
     * for the class named the same as the given class.
     * 
     * We do not use the session based getDescriptor() methods because they 
     * require the project be initialized with classes.  We are avoiding using 
     * a project with loaded classes so the project can be constructed prior to 
     * any class weaving.
     */
    public static ClassDescriptor findDescriptor(Project project, Class cls) {
        Iterator descriptors = project.getOrderedDescriptors().iterator();
        
        while (descriptors.hasNext()){
            ClassDescriptor descriptor = (ClassDescriptor) descriptors.next();
        
            if (descriptor.getJavaClassName().equals(cls.getName())){
                return descriptor;
            }
        }
        
        return null;
    }

    /**
     * INTERNAL:
     * Method to convert a getXyz or isXyz method name to an xyz attribute name. 
     */
    public static String getAttributeNameFromMethodName(String methodName) {
        String leadingChar, restOfName;
        
        if (methodName.startsWith(GET_PROPERTY_METHOD_PREFIX)) {
            leadingChar = methodName.substring(POSITION_AFTER_GET_PREFIX, POSITION_AFTER_GET_PREFIX + 1);
            restOfName = methodName.substring(POSITION_AFTER_GET_PREFIX + 1);
        } else {
            leadingChar = methodName.substring(POSITION_AFTER_IS_PREFIX, POSITION_AFTER_IS_PREFIX + 1);
            restOfName = methodName.substring(POSITION_AFTER_IS_PREFIX + 1);
        }
        
        return leadingChar.toLowerCase().concat(restOfName);
    }
    
    /**
     * INTERNAL:
     * Returns the same candidate methods as an entity listener would.
     */
    public static Method[] getCandidateCallbackMethodsForDefaultListener(MetadataEntityListener listener) {
        return getCandidateCallbackMethodsForEntityListener(listener);
    }
    
    /**
     * INTERNAL:
     * Return only the actual methods declared on this entity class.
     */
    public static Method[] getCandidateCallbackMethodsForEntityClass(Class entityClass) {
        return getDeclaredMethods(entityClass);
    }
    
    /**
     * INTERNAL:
     * Returns a list of methods from the given class, which can have private, 
     * protected, package and public access, AND will also return public 
     * methods from superclasses.
     */
    public static Method[] getCandidateCallbackMethodsForEntityListener(MetadataEntityListener listener) {
        HashSet candidateMethods = new HashSet();
        Class listenerClass = listener.getListenerClass();
        
        // Add all the declared methods ...
        Method[] declaredMethods = getDeclaredMethods(listenerClass);
        for (int i = 0; i < declaredMethods.length; i++) {
            candidateMethods.add(declaredMethods[i]);
        }
        
        // Now add any public methods from superclasses ...
        Method[] methods = getMethods(listenerClass);
        for (int i = 0; i < methods.length; i++) {
            if (candidateMethods.contains(methods[i])) {
                continue;
            }
            
            candidateMethods.add(methods[i]);
        }
        
        return (Method[]) candidateMethods.toArray(new Method[candidateMethods.size()]);
    }
    
    /**
     * INTERNAL:
     * Return potential lifecyle callback event methods for a mapped superclass. 
     * We must 'convert' the method to the entity class context before adding it 
     * to the listener.
     */
    public static Method[] getCandidateCallbackMethodsForMappedSuperclass(Class mappedSuperclass, Class entityClass) {
        ArrayList candidateMethods = new ArrayList();
        Method[] allMethods = getMethods(entityClass);
        Method[] declaredMethods = getDeclaredMethods(mappedSuperclass);
        
        for (int i = 0; i < declaredMethods.length; i++) {
            Method method = getMethodForName(allMethods, declaredMethods[i].getName());
            
            if (method != null) {
                candidateMethods.add(method);
            }
        }
        
        return (Method[]) candidateMethods.toArray(new Method[candidateMethods.size()]);
    }
    
    /**
     * INTERNAL:
     * Helper method that will return the class of the provided attribute.
     */
    public static Class getClassForAttribute(String attributeName, MetadataDescriptor mdd) {
        Class entityClass = mdd.getJavaClass();
        Field field = null;
        
        try {
            field = PrivilegedAccessController.getField(entityClass, attributeName, false);
        } catch (NoSuchFieldException nsfex) {
            throw ValidationException.unableToDetermineClassForAttribute(attributeName, entityClass.getName(), nsfex);
        }
        
        return field.getType();
    }

    /**
     * INTERNAL:
     * Load a class from a given class name
     */
    public static Class getClassForName(String classname, ClassLoader classLoader) {
        try {
    	    return PrivilegedAccessController.getClassForName(classname, true, classLoader);
        } catch (ClassNotFoundException exception) {
            throw ValidationException.unableToLoadClass(classname, exception);
        }
    }
    
    /**
     * INTERNAL:
	 * Get the declared methods from a class using the doPriveleged security
     * access. This call returns all methods (private, protected, package and
     * public) on the give class ONLY. It does not traverse the superclasses.
     */
	public static Method[] getDeclaredMethods(Class cls) {
        return PrivilegedAccessController.getDeclaredMethods(cls);
	}
    
    /**
     * INTERNAL:
     * Return the discriminator type class for the given discriminator type.
     */
    public static Class getDiscriminatorType(String discriminatorType) {
        if (m_discriminatorTypes == null) {
            m_discriminatorTypes = new HashMap();
            m_discriminatorTypes.put(MetadataConstants.CHAR, Character.class);
            m_discriminatorTypes.put(MetadataConstants.STRING, String.class);
            m_discriminatorTypes.put(MetadataConstants.INTEGER, Integer.class);
        }
        
        return (Class) m_discriminatorTypes.get(discriminatorType);
    }
    
    /**
     * INTERNAL:
     * Return the field classification for the given temporal type.
     */
    public static Class getFieldClassification(String temporalType) {
        if (m_fieldClassifications == null) {
            m_fieldClassifications = new HashMap();
            m_fieldClassifications.put(MetadataConstants.DATE, java.sql.Date.class);
            m_fieldClassifications.put(MetadataConstants.TIME, java.sql.Time.class);
            m_fieldClassifications.put(MetadataConstants.TIMESTAMP, java.sql.Timestamp.class);
        }
        
        return (Class) m_fieldClassifications.get(temporalType);
    }
    
    /**
     * INTERNAL:
     * Helper method that will return a given field based on the provided attribute name.
     */
    public static Field getFieldForAttribute(String attributeName, Class javaClass) {
        Field field = null;
        
        try {
            field = PrivilegedAccessController.getField(javaClass, attributeName, false);
        } catch (NoSuchFieldException nsfex) {
            return null;
        }
        
        return field;
    }

    /**
     * INTERNAL:
	 * Get the declared fields from a class using the doPriveleged security
     * access.
     */
	public static Field[] getFields(Class cls) {
        return PrivilegedAccessController.getDeclaredFields(cls);
	}  
    
    /**
     * INTERNAL:
     * Helper method to return a fully qualified column name. Will upper case
     * the complete column name.
     */
    public static String getFullyQualifiedColumnName(String name, String tableName) {
        String columnName = name;
        
        if (tableName != null && !tableName.equals("") && name.indexOf('.') < 0) {
            columnName = tableName + "." + name;
        }
        
        return columnName;
    }
    
    /**
     * INTERNAL:
     * Returns a fully qualified table name based on the values passed in.
     * eg. schema.catalog.name
     */
    public static String getFullyQualifiedTableName(String tableName, String catalog, String schema) {
        // catalog, attach it if specified
        if (! catalog.equals("")) {
            tableName = catalog + "." + tableName;
        }
        
        // schema, attach it if specified
        if (! schema.equals("")) {
            tableName = schema + "." + tableName;
        }
    
        return tableName;
    }
    
    /**
     * INTERNAL:
     * Returns a fully qualified table name based on the values passed in.
     * eg. schema.catalog.name
     */
    public static String getFullyQualifiedTableName(String name, String defaultName, String catalog, String schema) {
        // check if a name was specified otherwise use the default
        String tableName = name;
        if (tableName.equals("")) {
            tableName = defaultName;
        }
    
        return getFullyQualifiedTableName(tableName, catalog, schema);
    }
    
    /**
     * INTERNAL:
     * Method to return a generic method return type.
     */
	public static Type getGenericReturnType(Method method) {
        // WIP - should use PrivilegedAccessController
        return method.getGenericReturnType();
    }
    
    /**
     * INTERNAL:
     * Method to return a generic field type.
     */
	public static Type getGenericType(Field field) {
        // WIP - should use PrivilegedAccessController
        return field.getGenericType();
    }
    
    /**
     * INTERNAL:
	 * If the methodName passed in is a declared method on cls, then return
     * the methodName. Otherwise return null to indicate it does not exist.
	 */
    protected static Method getMethod(String methodName, Class cls, Class[] params) {
        try {
            return PrivilegedAccessController.getMethod(cls, methodName, params, true);
        } catch (NoSuchMethodException e1) {
            return null;
        }
    }
    
    /**
     * INTERNAL:
	 * If the methodName passed in is a declared method on cls, then return
     * the methodName. Otherwise return null to indicate it does not exist.
	 */
    protected static String getMethodName(String methodName, Class cls, Class[] params) {
        try {
            PrivilegedAccessController.getMethod(cls, methodName, params, true);
        } catch (NoSuchMethodException e1) {
            return null;
        }
        
        return methodName;
    }
    
    /**
     * INTERNAL:
	 */
    public static String[] getMethodNamesForField(String name, Object type) {
    	String getMethod = "";
    	String setMethod = "";
    	
    	// case #1:  field is boolean or Boolean prefixed with 'is', i.e. isOptional
    	//	- get method format = field name, i.e. isOptional()
    	//		- no work required
    	//	- set method format = setOptional()
    	//		- remove the 'is' portion and prefix with 'set'
    	
    	// case #2:  field is boolean or Boolean not prefixed with 'is', i.e. optional
    	//	- get method format = isOptional()
    	//		- capitalize the first letter and prefix with 'is'
    	//	- set method format = setOptional()
    	//		- capitalize the first letter and prefix with 'set'
    	
    	// case #3:  field is not boolean or Boolean, i.e. address
    	//	- get method format = getAddress()
    	//		- capitalize the first letter and prefix with 'get'
    	//	- set method format = setAddress()
    	//		- capitalize the first letter and prefix with 'set'
    	
    	if (type instanceof Boolean) {
    		if (name.startsWith(IS_PROPERTY_METHOD_PREFIX)) {
    			// case #1
    			getMethod = name;
    			setMethod = SET_PROPERTY_METHOD_PREFIX + name.substring(2);
    		} else {
    			// case #2
    			name = Character.toString(name.charAt(0)).toUpperCase() + name.substring(1);
    			getMethod = IS_PROPERTY_METHOD_PREFIX + name;
    			setMethod = SET_PROPERTY_METHOD_PREFIX + name;
    		}
    	} else {
    		// case #3
			name = Character.toString(name.charAt(0)).toUpperCase() + name.substring(1);
			getMethod = GET_PROPERTY_METHOD_PREFIX + name;
			setMethod = SET_PROPERTY_METHOD_PREFIX + name;
    	}
   	
        return new String[] {getMethod, setMethod}; 
    }
    
    /**
     * INTERNAL:
     * Find the method in the list where method.getName() == methodName.
     */
    public static Method getMethodForName(Method[] methods, String methodName) {
        for (int i = 0; i < methods.length; i++) {
            Method method = methods[i];
        
            if (method.getName().equals(methodName)) {
                return method;
            }
        }
        
        return null;
    }
    
    /**
     * INTERNAL:
	 * Get the methods from a class using the doPriveleged security access. 
     * This call returns only public methods from the given class and its 
     * superclasses.
     */
	public static Method[] getMethods(Class cls) {
        return PrivilegedAccessController.getMethods(cls);
	}
    
    /**
     * INTERNAL:
     * Return the raw class of the generic type.
     */
	public static Class getRawClassFromGeneric(Type type) {
        return (Class)(((ParameterizedType) type).getRawType());
    }
    
    /**
     * INTERNAL:
     * Method to return the correct reference class for either a method or a 
     * field. If targetEntity is not specified return the default.
     */
    public static Class getReferenceClass(Class defaultReferenceClass, Class targetEntity) {
        if (targetEntity == void.class) {
            return defaultReferenceClass;
        }
        
        return targetEntity;
    }
    
    /**
     * INTERNAL:
     * Get the method return type.
     */
	public static Class getReturnType(Method method) {
        return PrivilegedAccessController.getMethodReturnType(method);
    }
    
    /**
     * INTERNAL:
     * Helper method to return the type class of a ParameterizedType. This will 
     * handle the case for a generic collection. It now supports multiple types, 
     * e.g. Hastable<<String>, <Employee>>
     */
	public static Class getReturnTypeFromGeneric(Type type) {
        ParameterizedType pType = (ParameterizedType) type;
        
        if (java.util.Map.class.isAssignableFrom((Class) pType.getRawType())) {
            return (Class) pType.getActualTypeArguments()[1];
        }
        
        return (Class) pType.getActualTypeArguments()[0];
    }
    
    /**
     * INTERNAL:
	 * Method to convert a getMethod into a setMethod. This method could return 
     * null if the corresponding set method is not found.
	 */ 
    public static Method getSetMethod(Method method, Class cls) {
        String getMethodName = method.getName();
		Class[] params = new Class[] { method.getReturnType() };
            
        if (getMethodName.startsWith(GET_PROPERTY_METHOD_PREFIX)) {
            // Replace 'get' with 'set'.
            return getMethod(SET_PROPERTY_METHOD_PREFIX + getMethodName.substring(3), cls, params);
        }
        
        // methodName.startsWith(IS_PROPERTY_METHOD_PREFIX)
        // Check for a setXyz method first, if it exists use it.
        Method setMethod = getMethod(SET_PROPERTY_METHOD_PREFIX + getMethodName.substring(2), cls, params);
        
        if (setMethod == null) {
            // setXyz method was not found try setIsXyz
            return getMethod(SET_IS_PROPERTY_METHOD_PREFIX + getMethodName.substring(2), cls, params);
        }
        
        return setMethod;
	}
    
    /**
     * INTERNAL:
	 * Method to convert a getMethodName into a setMethodName. This method
     * could return null if the corresponding set method is not found.
	 */ 
    public static String getSetMethodName(Method method, Class cls) {
        String getMethodName = method.getName();
		Class[] params = new Class[] { method.getReturnType() };
            
        if (getMethodName.startsWith(GET_PROPERTY_METHOD_PREFIX)) {
            // Replace 'get' with 'set'.
            return getMethodName(SET_PROPERTY_METHOD_PREFIX + getMethodName.substring(3), cls, params);
        }
        
        // methodName.startsWith(IS_PROPERTY_METHOD_PREFIX)
        // Check for a setXyz method first, if it exists use it.
        String setMethodName = getMethodName(SET_PROPERTY_METHOD_PREFIX + getMethodName.substring(2), cls, params);
        
        if (setMethodName == null) {
            // setXyz method was not found try setIsXyz
            return getMethodName(SET_IS_PROPERTY_METHOD_PREFIX + getMethodName.substring(2), cls, params);
        }
        
        return setMethodName;
	}
    
    /**
     * INTERNAL:
     * Method to return a field type.
     */
	public static Class getType(Field field) {
        return PrivilegedAccessController.getFieldType(field);
    }
    
    /**
     * INTERNAL:
     * Method to return whether a collection type is a generic.
     */
	public static boolean isGenericCollectionType(Type type) {
        return (type instanceof ParameterizedType);
    }

    /**
     * INTERNAL:
     * Returns true is the given class is primitive wrapper type.
     */
     public static boolean isPrimitiveWrapperClass(Class cls) {
        return cls.equals(Long.class) ||
               cls.equals(Short.class) ||
               cls.equals(Float.class) ||
               cls.equals(Byte.class) ||
               cls.equals(Double.class) ||
               cls.equals(Number.class) ||
               cls.equals(Boolean.class) ||
               cls.equals(Integer.class) ||
               cls.equals(Character.class) ||
               cls.equals(java.math.BigInteger.class) ||
               cls.equals(java.math.BigDecimal.class);   
     }
     
    /**
     * INTERNAL:
     * Method to return whether a class is a supported Collection. EJB 3.0 spec 
     * currently only supports Collection and Set.
     */
	public static boolean isSupportedCollectionClass(Class cls) {
        return cls == Collection.class || 
               cls == Set.class || 
               cls == List.class || 
               cls == Map.class;
    }
    
    /**
     * INTERNAL:
     * Search the class for an attribute with a name matching 'attributeName'
     */
    public static boolean isValidAttributeName(String attributeName, Class javaClass) {
        Field attributes[] = getFields(javaClass);
        
        for (int i = 0; i < attributes.length; i++) {
            if (attributes[i].getName().equals(attributeName)) {
                return true;
            }
        }

        return false;
    }
    
    /**
     * INTERNAL:
     * Returns true if the given class is a valid clob type.
     */  
    public static boolean isValidClobType(Class cls) {
        return cls.equals(char[].class) ||
               cls.equals(String.class) ||
               cls.equals(Character[].class) ||
               cls.equals(java.sql.Clob.class);
    }
    
    /**
     * INTERNAL:
     * Returns true if the given class is a valid blob type.
     */ 
    public static boolean isValidBlobType(Class cls) {
        return cls.equals(byte[].class) ||
               cls.equals(Byte[].class) ||
               cls.equals(java.sql.Blob.class) ||
               Helper.classImplementsInterface(cls, java.io.Serializable.class);
    }
    
    /**
     * INTERNAL:
     * Return true if the given class is a valid enum type.
     */
    public static boolean isValidEnumeratedType(Class cls) {
        return cls.isEnum();    
    }
    
    /**
     * INTERNAL:
     * Returns true if the given class is a valid lob type.
     */
    public static boolean isValidLobType(Class cls) {
        return isValidClobType(cls) || isValidBlobType(cls);
    }
    
    /**
     * INTERNAL:
     */
    public static boolean isValidPersistenceMethodName(String methodName) {
        return methodName.startsWith(GET_PROPERTY_METHOD_PREFIX) || methodName.startsWith(IS_PROPERTY_METHOD_PREFIX);
    }
    
    /**
     * INTERNAL:
     * Returns true if the given class is valid for SerializedObjectMapping.
     */
    public static boolean isValidSerializedType(Class cls) {
        if (cls.isPrimitive()) {
            return false;
        }
        
        if (isPrimitiveWrapperClass(cls)) {    
            return false;
        }   
        
        if (isValidLobType(cls)) {
            return false;
        }
        
        if (isValidTemporalType(cls)) {
            return false;
        }
     
        return true;   
    }
     
    /**
     * INTERNAL:
     * Returns true if the given class is a valid temporal type and must be
     * marked temporal.
     */
     public static boolean isValidTemporalType(Class cls) {
        return (cls.equals(java.util.Date.class) ||
                cls.equals(java.util.Calendar.class) ||
                cls.equals(java.util.GregorianCalendar.class));
     }
     
    /** 
     * INTERNAL:
	 * Set the correct indirection policy on the collection mapping. Method
     * assume that the reference class has been set on the mapping before
     * calling this method.
	 */
	public static void setIndirectionPolicy(boolean usesIndirection, CollectionMapping mapping, Class rawClass, String mapKey) {
        if (usesIndirection) {
            if (rawClass == Map.class) {
                mapping.useTransparentMap(mapKey);
            } else if (rawClass == List.class) {
                mapping.useTransparentList();
            } else if (rawClass == Collection.class) {
                mapping.useTransparentCollection();
            } else if (rawClass == Set.class) {
                mapping.useTransparentSet();
            } else {
                // Because of validation we should never get this far.
            }
        } else {
            mapping.dontUseIndirection();
            
            if (rawClass == Map.class) {
                mapping.useMapClass(java.util.Hashtable.class, mapKey);
            } else {
                mapping.useCollectionClass(java.util.Vector.class);
            }
        }
    }
}
