/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2006, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.ejb.cmp3.transaction.base;

import javax.persistence.TransactionRequiredException;

import oracle.toplink.essentials.internal.ejb.cmp3.base.*;
import oracle.toplink.essentials.exceptions.TransactionException;
import oracle.toplink.essentials.internal.localization.ExceptionLocalization;
import oracle.toplink.essentials.internal.sessions.UnitOfWorkImpl;

/**
 * INTERNAL:
 * Base EntityTransactionWrapper
 * The EntityTransactionWrapper is used to make in transparent to an EntityManager
 * what kind of transaction is being used.  Transaction type can either be JTATransaction
 * or EntityTransaciton and they are mutually exclusive.  This is the implementation for
 * EntityTransaction
 * 
 * @see oracle.toplink.essentials.internal.ejb.cmp3.transaction.EntityTransactionWrapper
 * @see oracle.toplink.essentials.internal.ejb.cmp3.transaction.jdk14.EntityTransactionWrapper
 */
public class EntityTransactionWrapper extends TransactionWrapperImpl {

    protected EntityTransactionImpl entityTransaction;

    public EntityTransactionWrapper(EntityManagerImpl entityManager) {
        super(entityManager);
    }
    

    /**
     * INTERNAL:
     * This method will be used to check for a transaction and throws exception if none exists.
     * If this methiod returns without exception then a transaction exists.
     * This method must be called before accessing the localUOW.
     */
    public Object checkForTransaction(boolean validateExistence){
        if (entityTransaction != null && entityTransaction.isActive()) {
            return entityTransaction;
        }
        if (validateExistence){
            throwCheckTransactionFailedException();
        }
        return null;
    }

    /**
     * INTERNAL:
     * THis method is used to get the active UnitOfWork.  It is special in that it will
     * return the required RepeatableWriteUnitOfWork required by the EntityManager.  Once 
     * RepeatableWrite is merged into existing UnitOfWork this code can go away.
     */
    public RepeatableWriteUnitOfWork getTransactionalUnitOfWork(Object transaction){
        if (transaction == null){
            return null;
        }
        if (this.localUOW == null){
            this.localUOW = new RepeatableWriteUnitOfWork(entityManager.getServerSession().acquireClientSession());
        }
        return (RepeatableWriteUnitOfWork)this.localUOW;
    }
    
    public EntityManagerImpl getEntityManager(){
        return entityManager;
    }

    public void registerUnitOfWorkWithTxn(UnitOfWorkImpl uow){
        throw new TransactionRequiredException(ExceptionLocalization.buildMessage("join_trans_called_on_entity_trans"));// no JTA transactions availab
    }
    
    public boolean shouldClose() {
        return  (this.entityTransaction == null || ! this.entityTransaction.isActive());
    }

    protected void throwCheckTransactionFailedException() {
        throw TransactionException.transactionNotActive();
    }

    /**
     * These two method used for closing of EntityManager in case there is a transaction in progress:
     * The first method is called by EntityManager.close method to mark the current transaction,
     * the second one is called by EntityManager.verifyOpen method.
     */
    public void markLastTransaction() {
        if(entityTransaction != null) {
            entityTransaction.markLastTransaction();
        }
    }
    public boolean hasLastTransactionCompleted() {
        return entityTransaction != null && entityTransaction.hasLastTransactionCompleted();
    }

}
