/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2006, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.ejb.cmp3.xml;

import java.lang.reflect.Method;

/**
 * This class facilitates collection of default entity/listener class names 
 * and any callback method names that are to be applied.  This is useful in 
 * two-phase processing of the XML Descriptor, where the classloader may change 
 * between phases.  Here, phase one would involve gathering class/method name 
 * info and phase two would involve loading the classes, and setting the 
 * callback methods accordingly.
 * 
 * The annotations processor will also process the classes for extra callback
 * methods that are not specified in XML but decorated with an callback 
 * annotation.
 * 
 * @author Guy Pelletier, David McCann
 * @since EJB 3.0 Reference Implementation
 */
public class XMLEntityListener extends XMLListener {
	public XMLEntityListener(String listenerClassName, String entityClassName) {
		super(listenerClassName, entityClassName);
	}
    
    /**
     * INTERNAL:
     * Checks if the user already set this call back method via XML. We 
     * need to check because the set methods on the superclass will throw 
     * an exception when multiple methods are added for the callback.
     */
    private boolean noCallbackMethodAlreadySetFor(String event, Method method) {
        Method cbMethod = (Method) getMethods().get(event);
        
        if (cbMethod == null) {
            return true;
        } else {
            return ! cbMethod.getName().equals(method.getName());
        }
    }
    
    /**
     * INTERNAL:
     * Set it only if the same method wasn't already set from XML. If this
     * is a different method and one has already been set from XML, then an 
     * exception will be thrown from the set on the parent.
     */
    public void setPostBuildMethod(Method method) {
        if (noCallbackMethodAlreadySetFor(POST_BUILD, method)) {
            super.setPostBuildMethod(method);
        }
    }
    
    /**
     * INTERNAL:
     * Set it only if the same method wasn't already set from XML. If this
     * is a different method and one has already been set from XML, then an 
     * exception will be thrown from the set on the parent.
     */
    public void setPostCloneMethod(Method method) {
        if (noCallbackMethodAlreadySetFor(POST_CLONE, method)) {
            super.setPostCloneMethod(method);
        }
    }

    /**
     * INTERNAL:
     * Set it only if the same method wasn't already set from XML. If this
     * is a different method and one has already been set from XML, then an 
     * exception will be thrown from the set on the parent.
     */
    public void setPostDeleteMethod(Method method) {
        if (noCallbackMethodAlreadySetFor(POST_DELETE, method)) {
            super.setPostDeleteMethod(method);
        }
    }

    /**
     * INTERNAL:
     * Set it only if the same method wasn't already set from XML. If this
     * is a different method and one has already been set from XML, then an 
     * exception will be thrown from the set on the parent.
     */
    public void setPostInsertMethod(Method method) {
        if (noCallbackMethodAlreadySetFor(POST_INSERT, method)) {
            super.setPostInsertMethod(method);
        }
    }

    /**
     * INTERNAL:
     * Set it only if the same method wasn't already set from XML. If this
     * is a different method and one has already been set from XML, then an 
     * exception will be thrown from the set on the parent.
     */
    public void setPostRefreshMethod(Method method) {
        if (noCallbackMethodAlreadySetFor(POST_REFRESH, method)) {
            super.setPostRefreshMethod(method);
        }
    }
    
    /**
     * INTERNAL:
     * Set it only if the same method wasn't already set from XML. If this
     * is a different method and one has already been set from XML, then an 
     * exception will be thrown from the set on the parent.
     */
    public void setPostUpdateMethod(Method method) {
        if (noCallbackMethodAlreadySetFor(POST_UPDATE, method)) {
            super.setPostUpdateMethod(method);
        }
    }

    /**
     * INTERNAL:
     * Set it only if the same method wasn't already set from XML. If this
     * is a different method and one has already been set from XML, then an 
     * exception will be thrown from the set on the parent.
     */
    public void setPrePersistMethod(Method method) {
        if (noCallbackMethodAlreadySetFor(PRE_PERSIST, method)) {
            super.setPrePersistMethod(method);
        }
    }
    
    /**
     * INTERNAL:
     * Set it only if the same method wasn't already set from XML. If this
     * is a different method and one has already been set from XML, then an 
     * exception will be thrown from the set on the parent.
     */
    public void setPreRemoveMethod(Method method) {
        if (noCallbackMethodAlreadySetFor(PRE_REMOVE, method)) {
            super.setPreRemoveMethod(method);
        }
    }
    
    /**
     * INTERNAL:
     * Set it only if the same method wasn't already set from XML. If this
     * is a different method and one has already been set from XML, then an 
     * exception will be thrown from the set on the parent.
     */
    public void setPreUpdateWithChangesMethod(Method method) {
        if (noCallbackMethodAlreadySetFor(PRE_UPDATE_WITH_CHANGES, method)) {
            super.setPreUpdateWithChangesMethod(method);
        }
    }
}
