/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2005, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.helper;

import oracle.toplink.essentials.exceptions.ValidationException;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Serializable;

/**
 * <p>Provide common functionalities for serialization of object.
 * </p>
 *
 * <p>This class throws exceptions for invalid <code>null</code> inputs.
 * Each method documents its behaviour in more detail.</p>
 *
 * @author Steven Vo
 * @since OracleAS 10.0.3
 */
public class SerializationHelper {

    /**
     * <p>Deep clone a Serializable object using serialization.
     * @param the serializable object
     * @return the deep cloned object
     * @throws  IOException, ClassNotFoundException
     */
    public static Object clone(Serializable object) throws IOException, ClassNotFoundException {
        return deserialize(serialize(object));
    }

    /**
     * Serialize the object to an OutputStream
     *
     * @param obj  the object to serialize to bytes
     * @param outputStream  the stream to write to, can not be null
     * @throws IOException
     */
    public static void serialize(Serializable obj, OutputStream outputStream) throws IOException {
        if (outputStream == null) {
            throw ValidationException.invalidNullMethodArguments();
        }
        ObjectOutputStream outStream = null;

        try {
            // stream closed in the finally
            outStream = new ObjectOutputStream(outputStream);
            outStream.writeObject(obj);
        } finally {
            try {
                if (outStream != null) {
                    outStream.close();
                }
            } catch (IOException ex) {
                // ignore;
            }
        }
    }

    /**
     * Serialize the object to a byte array
     *
     * @param obj  the object to serialize to bytes
     * @return a byte[] of the obj
     * @throws IOException
     */
    public static byte[] serialize(Serializable obj) throws IOException {
        ByteArrayOutputStream outStream = new ByteArrayOutputStream(512);
        serialize(obj, outStream);
        return outStream.toByteArray();
    }

    /**
     * Deserialze an object from an InputStream
     *
     * @param inputStream  the serialized object input stream, must not be null
     * @return the deserialized object
     * @throws IOException, ClassNotFoundException
     */
    public static Object deserialize(InputStream inputStream) throws IOException, ClassNotFoundException {
        if (inputStream == null) {
            throw new IllegalArgumentException("The inputStream argument cannot be null");
        }
        ObjectInputStream inStream = null;
        try {
            // stream closed in the finally
            inStream = new ObjectInputStream(inputStream);
            return inStream.readObject();

        } finally {
            try {
                if (inStream != null) {
                    inStream.close();
                }
            } catch (IOException ex) {
                // ignore
            }
        }
    }

    /**
     * Deserialize an object from a byte array
     *
     * @param objectBytes  the serialized object, can not be null
     * @return the deserialized object
     * @throws  IOException, ClassNotFoundException
     */
    public static Object deserialize(byte[] objectBytes) throws IOException, ClassNotFoundException {
        if (objectBytes == null) {
            throw ValidationException.invalidNullMethodArguments();
        }
        ByteArrayInputStream inStream = new ByteArrayInputStream(objectBytes);
        return deserialize(inStream);
    }
}
