/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2005, Oracle. All rights reserved.  
package oracle.toplink.essentials.logging;


/** Java class "JavaLog.java" generated from Poseidon for UML.
 *  Poseidon for UML is developed by <A HREF="http://www.gentleware.com">Gentleware</A>.
 *  Generated with <A HREF="http://jakarta.apache.org/velocity/">velocity</A> template engine.
 */
import java.util.*;
import java.util.logging.*;
import oracle.toplink.essentials.sessions.Session;
import oracle.toplink.essentials.internal.localization.i18n.*;

/**
 * PUBLIC:
 * <p>
 * This is a wrapper class for java.util.logging.  It is used when messages need to
 * be logged through java.util.logging.
 * </p>
 *  @see SessionLog
 *  @see AbstractSessionLog
 *  @see SessionLogEntry
 *  @see Session
 */
public class JavaLog extends AbstractSessionLog {

    /**
     * Represents the Hashtable that stores all the name space strings.
     * The keys are category names.  The values are namespace strings.
     */
    private Hashtable nameSpaceTable;

    /**
     * Represents the Hashtable that stores all the loggers.
     * The keys are category names.  The values are loggers.
     */
    private Hashtable loggerTable;

    /**
     * Represents the Hashtable that stores booleans on if each category should be logged.
     * The keys are category names.  The values are shouldLog booleans.
     */
    private Hashtable shouldLogTable;

    /**
     * Stores all the java.util.logging.Levels.  The indexes are TopLink logging levels.
     */
    private Level[] levels = new Level[] { Level.ALL, Level.FINEST, Level.FINER, Level.FINE, Level.CONFIG, Level.INFO, Level.WARNING, Level.SEVERE, Level.OFF };

    /**
     * Stores the number of elements in levels.
     */
    private int elementCount = 9;

    /**
     * Stores the namespace for session, i.e."oracle.toplink.essentials.<sessionname>".
     */
    private String sessionNameSpace;

    /**
     * Stores the Logger for default TopLink namespace, i.e. "oracle.toplink.essentials".
     */
    private Logger defaultTopLinkLogger;

    /**
     * Stores the Logger for session namespace, i.e. "oracle.toplink.essentials.<sessionname>".
     */
    private Logger sessionLogger;

    /**
     * Stores shouldLog for default TopLink namespace, i.e. "oracle.toplink.essentials".
     */
    private Boolean defaultTopLinkShouldLog;

    /**
     * Stores shouldLog for session namespace, i.e. "oracle.toplink.essentials.<sessionname>".
     */
    private Boolean sessionShouldLog;

    /**
     * Stores the default session name in case there is the session name is missing.
     */
    protected static final String DEFAULT_SESSION_NAME = "defaultsessionname";
    public static final String DEFAULT_TOPLINK_NAMESPACE = "oracle.toplink.essentials";
    protected static final String LOGGING_LOCALIZATION_STRING = "oracle.toplink.essentials.internal.localization.i18n.LoggingLocalizationResource";
    protected static final String TRACE_LOCALIZATION_STRING = "oracle.toplink.essentials.internal.localization.i18n.TraceLocalizationResource";

    /**
     * PUBLIC:
     * <p>
     * Create and return a new JavaLog.
     * </p>
     */
    public JavaLog() {
        super();
        initializeHashtables();
    }

    /**
     * <p>
     * Create all three hash tables.
     * </p>
     */
    protected void initializeHashtables() {
        nameSpaceTable = new Hashtable();
        loggerTable = new Hashtable();
        shouldLogTable = new Hashtable();
    }

    /**
     * PUBLIC:
     * <p>
     * Return the effective log level for the name space extracted from session and category.
     * If a Logger's level is set to be null then the Logger will use an effective Level that will
     * be obtained by walking up the parent tree and using the first non-null Level.
     * </p><p>
     *
     * @return the effective log level.
     * </p>
     */
    public int getLevel(String category) {
        Logger logger = getLogger(category);
        while ((logger != null) && (logger.getLevel() == null)) {
            logger = logger.getParent();
        }

        if (logger == null) {
            return OFF;
        }

        //For a given java.util.logging.Level, return the index (ie, TopLink logging level)
        int logLevel = logger.getLevel().intValue();
        for (int i = 0; i < elementCount; i++) {
            if (logLevel == levels[i].intValue()) {
                return i;
            }
        }
        return OFF;
    }

    /**
     * PUBLIC:
     * <p>
     * Set the log level to a logger with name space extracted from the given category.
     * </p>
     */
    public void setLevel(int level, String category) {
        Logger logger = getLogger(category);
        logger.setLevel(getJavaLevel(level));

        //shouldLog cache should be updated based on the new level.
        Boolean shouldLog = new Boolean(logger.isLoggable(getJavaLevel(level)));

        if (session == null) {
            defaultTopLinkShouldLog = shouldLog;
            return;
        }

        if ((category == null) || (category.length() == 0)) {
            sessionShouldLog = shouldLog;
            return;
        }

        shouldLogTable.put(category, shouldLog);
    }

    /**
     * INTERNAL:
     * Return the name space for the given category from the hashtable.  If not found,
     * create a new one and store it in the hashtable.
     */
    public String getNameSpaceString(String category) {
        if (session == null) {
            return DEFAULT_TOPLINK_NAMESPACE;
        }

        if ((category == null) || (category.length() == 0)) {
            return sessionNameSpace;
        }

        String nameSpaceString = (String)nameSpaceTable.get(category);
        if (nameSpaceString == null) {
            nameSpaceString = sessionNameSpace + "." + category;
            nameSpaceTable.put(category, nameSpaceString);
        }
        return nameSpaceString;
    }

    /**
     * INTERNAL:
     * Return the Logger for the given category from the hashtable.  If not found,
     * create a new one and store it in the hashtable.
     */
    public Logger getLogger(String category) {
        if (session == null) {
            if (defaultTopLinkLogger == null) {
                defaultTopLinkLogger = Logger.getLogger(DEFAULT_TOPLINK_NAMESPACE);
            }
            return defaultTopLinkLogger;
        }

        if ((category == null) || (category.length() == 0)) {
            if (sessionLogger == null) {
                sessionLogger = Logger.getLogger(sessionNameSpace);
            }
            return sessionLogger;
        }

        Logger logger = (Logger)loggerTable.get(category);
        if (logger == null) {
            logger = Logger.getLogger(getNameSpaceString(category));
            loggerTable.put(category, logger);
        }
        return logger;
    }

    /**
     * PUBLIC:
     * <p>
     * Set the session and session namespace.
     * </p>
     *
     * @param session  a Session
     * </p>
     */
    public void setSession(Session session) {
        super.setSession(session);
        if (session != null) {
            String sessionName;
            if ((session.getName() != null) && (session.getName().length() != 0)) {
                sessionName = session.getName();
            } else {
                sessionName = DEFAULT_SESSION_NAME;
            }
            sessionNameSpace = DEFAULT_TOPLINK_NAMESPACE + "." + sessionName;
        }
    }

    /**
     * INTERNAL:
     * Return the corresponding java.util.logging.Level for a given TopLink level.
     */
    private Level getJavaLevel(int level) {
        return levels[level];
    }

    /**
     * PUBLIC:
     * <p>
     * Check if a message of the given level would actually be logged by the logger
     * with name space built from the given session and category.
     * Return the shouldLog for the given category from the hashtable.  If not found,
     * create a new one and store it in the hashtable.
     * </p><p>
     * @return true if the given message level will be logged
     * </p>
     */
    public boolean shouldLog(int level, String category) {
        if (session == null) {
            if (defaultTopLinkShouldLog == null) {
                defaultTopLinkShouldLog = new Boolean(getLogger(DEFAULT_TOPLINK_NAMESPACE).isLoggable(getJavaLevel(level)));
            }
            return defaultTopLinkShouldLog.booleanValue();
        }

        if ((category == null) || (category.length() == 0)) {
            if (sessionShouldLog == null) {
                sessionShouldLog = new Boolean(getLogger(sessionNameSpace).isLoggable(getJavaLevel(level)));
            }
            return sessionShouldLog.booleanValue();
        }

        Boolean shouldLog = (Boolean)shouldLogTable.get(category);
        if (shouldLog == null) {
            shouldLog = new Boolean(getLogger(category).isLoggable(getJavaLevel(level)));
            shouldLogTable.put(category, shouldLog);
        }
        return shouldLog.booleanValue();
    }

    /**
     * PUBLIC:
     * <p>
     * Log a SessionLogEntry
     * </p><p>
     * @param entry SessionLogEntry that holds all the information for a TopLink logging event
     * </p>
     */
    public void log(SessionLogEntry entry) {
        if (!shouldLog(entry.getLevel(), entry.getNameSpace())) {
            return;
        }

        Logger logger = getLogger(entry.getNameSpace());
        Level javaLevel = getJavaLevel(entry.getLevel());

        internalLog(entry, javaLevel, logger);
    }

    /**
     * INTERNAL:
     * <p>
     * Build a LogRecord
     * </p><p>
     * @param entry SessionLogEntry that holds all the information for a TopLink logging event
     * @param javaLevel the message level
     * </p>
     */
    protected void internalLog(SessionLogEntry entry, Level javaLevel, Logger logger) {
        TopLinkLogRecord lr = new TopLinkLogRecord(javaLevel, entry.getMessage());
        lr.setSourceClassName(null);
        lr.setSourceMethodName(null);
        lr.setParameters(entry.getParameters());
        lr.setLoggerName(getNameSpaceString(entry.getNameSpace()));
        if (entry.shouldTranslate()) {
            String bundleName;
            ResourceBundle resourceBundle;
            if (entry.getLevel() > FINE) {
                bundleName = LOGGING_LOCALIZATION_STRING;
                resourceBundle = new LoggingLocalizationResource();
            } else {
                bundleName = TRACE_LOCALIZATION_STRING;
                resourceBundle = new TraceLocalizationResource();
            }
            lr.setResourceBundleName(bundleName);
            lr.setResourceBundle(resourceBundle);
        }
        if (shouldPrintSession()) {
            lr.setSessionString(getSessionString(entry.getSession()));
        }
        if (shouldPrintConnection()) {
            lr.setConnection(entry.getConnection());
        }
        lr.setThrown(entry.getException());
        lr.setShouldLogExceptionStackTrace(shouldLogExceptionStackTrace());
        lr.setShouldPrintDate(shouldPrintDate());
        lr.setShouldPrintThread(shouldPrintThread());
        logger.log(lr);
    }

    /**
     * PUBLIC:
     * <p>
     * Log a throwable.
     * </p><p>
     * @param throwable a throwable
     * </p>
     */
    public void throwing(Throwable throwable) {
        getLogger(null).throwing(null, null, throwable);
    }

    /**
     * INTERNAL:
     * Each session owns its own session log because session is stored in the session log
     */
    public Object clone() {
        JavaLog cloneLog = (JavaLog)super.clone();
        if (cloneLog != null) {
            cloneLog.initializeHashtables();
            return cloneLog;
        }
        return null;
    }
}
